/* **********************************************************
 * Copyright 2012-2013, 2019, 2020 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

/*
 * ProviderDefinition.java --
 *
 *      Detailed information about a vAPI provider.
 */
package com.vmware.vapi.core;

import com.vmware.vapi.internal.util.Validate;

/**
 * The <code>ProviderDefinition</code> contains detailed information about
 * vAPI provider.
 */
public class ProviderDefinition {
    private final String id;
    private final String checkSum;

    /**
     * Constructor.
     *
     * @param id  identifier of the provider
     * @param checkSum checksum of the provider's API
     *
     * @see #getCheckSum()
     */
    public ProviderDefinition(String id, String checkSum) {
        Validate.notNull(id);
        Validate.notNull(checkSum);
        this.id = id;
        this.checkSum = checkSum;
    }

    /**
     * Returns the identifier of the provider.
     *
     * @return  <code>String</code> representing the id
     */
    public String getIdentifier() {
        // TODO: all other xxxDefinitions has xxxIdentifier
        //       is it OK to go with String here?
        return id;
    }

    /**
     * Returns a checksum for the introspected API. Checksum is computed on
     * <ul>
     * <li>API Service identifiers
     * <li>API operation identifiers
     * <li>API operation details: input, output and error definitions
     *
     * <p>Result is <tt>CRC-32</tt> checksum as a HEX-encoded <code>String</code>.
     *
     * @return the computed checksum
     */
    public String getCheckSum() {
        return checkSum;
    }

    @Override
    public String toString() {
        return "Provider id: " + id;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof ProviderDefinition)) {
            return false;
        }
        ProviderDefinition other = (ProviderDefinition) obj;
        return getIdentifier().equals(other.getIdentifier())
                && getCheckSum().equals(other.getCheckSum());
    }

    @Override
    public int hashCode() {
        return getIdentifier().hashCode();
    }

}

