/* **********************************************************
 * Copyright 2013-2014 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.data;

import java.util.EnumSet;
import java.util.LinkedList;
import java.util.List;

import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;


/**
 * <code>DataDefinition</code> for dynamic structures.
 */
public class DynamicStructDefinition extends DataDefinition {
    private static final DynamicStructDefinition INSTANCE =
            new DynamicStructDefinition();

    private static final EnumSet<DataType> VALID_DATA_TYPES =
            EnumSet.of(DataType.STRUCTURE, DataType.ERROR);

    public static DynamicStructDefinition getInstance() {
        return INSTANCE;
    }

    /**
     * Constructor.
     *
     * <p>Private one, since this class is a Singleton.
     */
    private DynamicStructDefinition() {
    }

    @Override
    public DataType getType() {
        return DataType.DYNAMIC_STRUCTURE;
    }

    /**
     * Validates that the specified {@link DataValue} is an instance of this
     * data definition.
     *
     * <p>Validates that supplied <code>value</code> is not <code>null</code>
     * and it's type matches {@link DataType#STRUCTURE}.
     *
     * @param value  the <code>DataValue</code> to validate
     * @return       a list of messages describing validation problems; empty
     *               list indicates that validation was successful
     */
    @Override
    public List<Message> validate(DataValue value) {
        List<Message> result = new LinkedList<Message>();
        if (value == null) {
            result.add(MessageFactory.getMessage("vapi.data.dynamicstruct.validate.mismatch",
                                                 VALID_DATA_TYPES.toString(),
                                                 "null"));
        } else if (!VALID_DATA_TYPES.contains(value.getType())) {
            result.add(MessageFactory.getMessage("vapi.data.dynamicstruct.validate.mismatch",
                                                 VALID_DATA_TYPES.toString(),
                                                 value.getType().toString()));
        }
        return result;
    }

    @Override
    public void accept(DefinitionVisitor visitor) {
        visitor.visit(this);
    }
}
