/* **********************************************************
 * Copyright (c) 2011-2012, 2016 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.data;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import com.vmware.vapi.CoreException;

/**
 * A list of values.
 */
public final class ListValue implements DataValue, Iterable<DataValue> {

    private static final long serialVersionUID = 1L;
    private final List<DataValue> list;

    /**
     * Creates an empty list value.
     */
    public ListValue() {
        this.list = new LinkedList<>();
    }

    @Override
    public DataType getType() {
        return DataType.LIST;
    }

    /**
     * Appends the specified element to the end of this list.
     *
     * @param value  data value
     */
    public void add(DataValue value) {
        list.add(value);
    }

    /**
     * Appends all of the values in the specified collection to the end of this
     * list, in the order that they are returned by the specified collection's
     * iterator.
     *
     * @param c  collection of values
     */
    public void addAll(Collection<? extends DataValue> c) {
        list.addAll(c);
    }

    @Override
    public boolean equals(Object o) {
        if (o instanceof ListValue) {
            ListValue otherList = (ListValue) o;
            return list.equals(otherList.list);
        }
        return false;
    }

    /**
     * Returns the value at the specified position.
     *
     * @param index  index of the value to return
     * @return the value at the specified position
     */
    public DataValue get(int index) {
        return list.get(index);
    }

    @Override
    public int hashCode() {
        return list.hashCode();
    }

    /**
     * Returns <code>true</code> if this list contains no values.
     *
     * @return whether the list is empty
     */
    public boolean isEmpty() {
        return list.isEmpty();
    }

    @Override
    public Iterator<DataValue> iterator() {
        return Collections.unmodifiableList(list).iterator();
    }

    /**
     * Returns the number of values in this list.
     *
     * @return the number of values in this list
     */
    public int size() {
        return list.size();
    }

    @Override
    public String toString() {
        return list.toString();
    }

    /**
     * Returns read-only view of the underlying data.
     *
     * @return view of the underlying data as standard list
     */
    public List<DataValue> getList() {
        return Collections.unmodifiableList(list);
    }

    public List<Long> getIntegerList() {
        List<Long> intList = new LinkedList<>();
        for (DataValue dv : this.list) {
            if (dv instanceof OptionalValue) {
                intList.add(((OptionalValue) dv).getIntegerOrNull());
            } else if (dv instanceof IntegerValue) {
                intList.add(((IntegerValue) dv).getValue());
            } else {
                throw new CoreException("vapi.data.list.get.mismatch",
                                        "Integer",
                                        dv.getType().toString());
            }
        }
        return intList;
    }

    public List<Double> getDoubleList() {
        List<Double> dblList = new LinkedList<>();
        for (DataValue dv : this.list) {
            if (dv instanceof OptionalValue) {
                dblList.add(((OptionalValue) dv).getDoubleOrNull());
            } else if (dv instanceof DoubleValue) {
                dblList.add(((DoubleValue) dv).getValue());
            } else {
                throw new CoreException("vapi.data.list.get.mismatch",
                                        "Double",
                                        dv.getType().toString());
            }
        }
        return dblList;
    }

    public List<Boolean> getBooleanList() {
        List<Boolean> boolList = new LinkedList<>();
        for (DataValue dv : this.list) {
            if (dv instanceof OptionalValue) {
                boolList.add(((OptionalValue) dv).getBooleanOrNull());
            } else if (dv instanceof BooleanValue) {
                boolList.add(((BooleanValue) dv).getValue());
            } else {
                throw new CoreException("vapi.data.list.get.mismatch",
                                        "Boolean",
                                        dv.getType().toString());
            }
        }
        return boolList;
    }

    /**
     *
     * @return
     * @throws CoreException if this list contains value which type differs
     *                       from <code>StringValue</code> or <code>OptionalValue</code>
     *                       of <code>StringValue</code>.
     */
    public List<String> getStringList() {
        List<String> strList = new LinkedList<>();
        for (DataValue dv : this.list) {
            if (dv instanceof OptionalValue) {
                strList.add(((OptionalValue) dv).getStringOrNull());
            } else if (dv instanceof StringValue) {
                strList.add(((StringValue) dv).getValue());
            } else {
                throw new CoreException("vapi.data.list.get.mismatch",
                                        "String",
                                        dv.getType().toString());
            }
        }
        return strList;
    }

    public List<char[]> getSecretList() {
        List<char[]> secretList = new LinkedList<>();
        for (DataValue dv : this.list) {
            if (dv instanceof OptionalValue) {
                secretList.add(((OptionalValue) dv).getSecretOrNull());
            } else if (dv instanceof SecretValue) {
                secretList.add(((SecretValue) dv).getValue());
            } else {
                throw new CoreException("vapi.data.list.get.mismatch",
                                        "Secret",
                                        dv.getType().toString());
            }
        }
        return secretList;
    }

    /*
     * A whole bunch of factory methods for creating ListValue instances from
     * Java List<?> instances.
     */

    public static ListValue integerList(List<Long> intList) {
        ListValue listValue = new ListValue();
        for (Long val : intList) {
            listValue.add(new IntegerValue(val.longValue()));
        }
        return listValue;
    }

    public static ListValue
            doubleList(List<Double> dblList) {
        ListValue listValue = new ListValue();
        for (Double val : dblList) {
            listValue.add(new DoubleValue(val.doubleValue()));
        }
        return listValue;
    }

    public static ListValue
            booleanList(List<Boolean> boolList) {
        ListValue listValue = new ListValue();
        for (Boolean val : boolList) {
            listValue.add(BooleanValue.getInstance(val.booleanValue()));
        }
        return listValue;
    }

    public static ListValue
            stringList(List<String> strList) {
        ListValue listValue = new ListValue();
        for (String val : strList) {
            listValue.add(new StringValue(val));
        }
        return listValue;
    }

    public static ListValue
            secretList(List<char[]> secretList) {
        ListValue listValue = new ListValue();
        for (char[] val : secretList) {
            listValue.add(new SecretValue(val));
        }
        return listValue;
    }

    public static ListValue optionalIntegerList(List<Long> intList) {
        ListValue listValue = new ListValue();
        for (Long val : intList) {
            listValue.add(new OptionalValue(val));
        }
        return listValue;
    }

    public static ListValue optionalDoubleList(List<Double> dblList) {
        ListValue listValue = new ListValue();
        for (Double val : dblList) {
            listValue.add(new OptionalValue(val));
        }
        return listValue;
    }

    public static ListValue optionalBooleanList(List<Boolean> boolList) {
        ListValue listValue = new ListValue();
        for (Boolean val : boolList) {
            listValue.add(new OptionalValue(val));
        }
        return listValue;
    }

    public static ListValue optionalStringList(List<String> strList) {
        ListValue listValue = new ListValue();
        for (String val : strList) {
            listValue.add(new OptionalValue(val));
        }
        return listValue;
    }

    @Override
    public void accept(ValueVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public ListValue copy() {
        ListValue result = new ListValue();
        for (DataValue element : this.list) {
            result.add(element.copy());
        }
        return result;
    }
}
