/* **********************************************************
 * Copyright (c) 2011-2014, 2016, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.data;


import java.util.List;

import com.vmware.vapi.*;
import com.vmware.vapi.internal.util.Validate;

/**
 * An <code>OptionalDefinition</code> instance defines an optional type with
 * a specific element type.
 */
public class OptionalDefinition extends DataDefinition {
    /**
     * The type of the element that is optional.
     */
    private DataDefinition elementType;

    /**
     * Constructor to create a optional-definition with the specified element
     * type.
     *
     * @param elementType  data-definition for the element of the
     *                     optional-definition
     *
     * @throws IllegalArgumentException if {@code elementType} is {@code null}
     */
    public OptionalDefinition(DataDefinition elementType) {
        Validate.notNull(elementType,
                         "Optional definition requires element definition.");
        this.elementType = elementType;
    }

    @Override
    public DataType getType() {
        return DataType.OPTIONAL;
    }

    /**
     * Create and return a new unset {@link OptionalValue}.
     *
     * @return an empty optional value.
     */
    public OptionalValue newInstance() {
        return OptionalValue.UNSET;
    }

    /**
     * Return the data-definition for the element type of this
     * optional-definition.
     *
     * @return  the data-definition for the element type of this
     *          optional-definition
     */
    public DataDefinition getElementType() {
        return this.elementType;
    }

    /**
     * {@inheritDoc}
     *
     * <p>In addition, validates the type of the optional value content
     * if it is set/present.
     */
    @Override
    public List<Message> validate(DataValue value) {
        List<Message> errors = super.validate(value);

        if (!errors.isEmpty()) {
            return errors;
        }

        // this is safe because super.validate() verified the type
        OptionalValue optVal = (OptionalValue) value;

        if (optVal.isSet()) {
            List<Message> subErrors = getElementType().validate(optVal.getValue());
            if (!subErrors.isEmpty()) {
                errors.add(MessageFactory.getMessage("vapi.data.optional.validate"));
                errors.addAll(subErrors);
            }
        } else {
            // nulls are valid for optional so this is a
            // no-op for a good reason
        }

        return errors;
    }

    @Override
    public void completeValue(DataValue value) {
        if (value != null && value.getType() == DataType.OPTIONAL) {
            OptionalValue optValue = (OptionalValue) value;
            elementType.completeValue(optValue.getValue());
        }
    }

    @Override
    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }

        if (!(other instanceof OptionalDefinition)) {
            return false;
        }
        OptionalDefinition otherOptional = (OptionalDefinition) other;
        return this.elementType.equals(otherOptional.elementType);
    }

    @Override
    public int hashCode() {
        return 17 + 31 * elementType.hashCode();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        return "Optional[" + elementType.toString() + "]";
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void accept(DefinitionVisitor visitor) {
        visitor.visit(this);
    }

}
