/* **********************************************************
 * Copyright (c) 2011-2013, 2016 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.data;

import com.vmware.vapi.CoreException;

public final class OptionalValue implements DataValue {

    /**
     * {@code OptionalValue} instance which is not set, i.e. {@link #getValue()}
     * is guaranteed to return {@code null}.
     *
     * <p>This immutable instance can be safely used in any context to avoid
     * excessive class instantiation.
     */
    public static final OptionalValue UNSET = new OptionalValue();

    private static final long serialVersionUID = 1L;
    private DataValue value;

    private void init(DataValue value) {
        this.value = value;
    }

    public OptionalValue() {
        init(null);
    }

    public OptionalValue(DataValue value) {
        init(value);
    }

    // Convenience constructors for creating optional primitive values

    public OptionalValue(boolean value) {
        init(BooleanValue.getInstance(value));
    }

    public OptionalValue(long value) {
        init(new IntegerValue(value));
    }

    public OptionalValue(double value) {
        init(new DoubleValue(value));
    }

    public OptionalValue(String value) {
        if (value == null) {
            init(null);
        } else {
            init(new StringValue(value));
        }
    }

    public OptionalValue(char[] value) {
        if (value == null) {
            init(null);
        } else {
            // secret value does a copy of the value
            init(new SecretValue(value));
        }
    }

    public OptionalValue(Boolean value) {
        if (value == null) {
            init(null);
        } else {
            init(BooleanValue.getInstance(value.booleanValue()));
        }
    }

    public OptionalValue(Long value) {
        if (value == null) {
            init(null);
        } else {
            init(new IntegerValue(value.longValue()));
        }
    }

    public OptionalValue(Double value) {
        if (value == null) {
            init(null);
        } else {
            init(new DoubleValue(value.doubleValue()));
        }
    }

    @Override
    public DataType getType() {
        return DataType.OPTIONAL;
    }

    public boolean isSet() {
        return value != null;
    }

    /**
     * @return the content of this {@code OptionalValue} if it holds one
     *         or {@code null} if it is not set.
     */
    public DataValue getValue() {
        return this.value;
    }

    @Override
    public boolean equals(Object o) {

        if (o instanceof OptionalValue) {
            OptionalValue opt = (OptionalValue) o;
            if (isSet() && opt.isSet()) {
                return value.equals(opt.value);
            } else if (!isSet() && !opt.isSet()) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        if (value != null) {
            return value.hashCode();
        }
        return 0;
    }

    @Override
    public String toString() {
        if (value == null) {
            return "<unset>";
        } else {
            return value.toString();
        }
    }

    //
    // Convenience methods below
    //

    private long getIntegerInternal(boolean throwOnUnset, long orElse) {
        DataValue value = getValue();

        if (isSet()) {
            if (value instanceof IntegerValue) {
                IntegerValue tmp = (IntegerValue) value;
                return tmp.getValue();
            } else {
                throw new CoreException("vapi.data.optional.getvalue.mismatch",
                                        "Integer",
                                        value.getType().toString());
            }
        } else if (throwOnUnset) {
            throw new CoreException("vapi.data.optional.getvalue.unset");
        }
        return orElse;
    }

    public long getInteger() {
        return getIntegerInternal(true, 0);
    }

    public long getIntegerOrElse(long orElse) {
        return getIntegerInternal(false, orElse);
    }

    public Long getIntegerOrNull() {
        return isSet() ? new Long(getInteger()) : null;
    }

    private double getDoubleInternal(boolean throwOnUnset, double orElse) {
        DataValue value = getValue();

        if (isSet()) {
            if (value instanceof DoubleValue) {
                DoubleValue tmp = (DoubleValue) value;
                return tmp.getValue();
            } else {
                throw new CoreException("vapi.data.optional.getvalue.mismatch",
                                        "Double",
                                        value.getType().toString());
            }
        } else if (throwOnUnset) {
            throw new CoreException("vapi.data.optional.getvalue.unset");

        }
        return orElse;
    }

    public double getDouble() {
        return getDoubleInternal(true, 0);
    }

    public double getDoubleOrElse(double orElse) {
        return getDoubleInternal(false, orElse);
    }

    public Double getDoubleOrNull() {
        return isSet() ? new Double(getDouble()) : null;
    }

    private boolean getBooleanInternal(boolean throwOnUnset, boolean orElse) {
        DataValue value = getValue();

        if (isSet()) {
            if (value instanceof BooleanValue) {
                BooleanValue tmp = (BooleanValue) value;
                return tmp.getValue();
            } else {
                throw new CoreException("vapi.data.optional.getvalue.mismatch",
                                        "Boolean",
                                        value.getType().toString());
            }
        } else if (throwOnUnset) {
            throw new CoreException("vapi.data.optional.getvalue.unset");
        }
        return orElse;
    }

    public boolean getBoolean() {
        return getBooleanInternal(true, true);
    }

    public boolean getBooleanOrElse(boolean orElse) {
        return getBooleanInternal(false, orElse);
    }

    public Boolean getBooleanOrNull() {
        return isSet() ? Boolean.valueOf(getBoolean()) : null;
    }

    private String
            getStringInternal(boolean throwOnUnset, String orElse) {
        DataValue value = getValue();

        if (isSet()) {
            if (value instanceof StringValue) {
                return ((StringValue) value).getValue();
            } else {
                throw new CoreException("vapi.data.optional.getvalue.mismatch",
                                        "String",
                                        value.getType().toString());
            }
        } else if (throwOnUnset) {
            throw new CoreException("vapi.data.optional.getvalue.unset");
        }
        return orElse;
    }

    public String getString() {
        return getStringInternal(true, "");
    }

    public String getStringOrElse(String orElse) {
        return getStringInternal(false, orElse);
    }

    public String getStringOrNull() {
        return getStringOrElse(null);
    }

    private char[]
            getSecretInternal(boolean throwOnUnset, char[] orElse) {
        DataValue value = getValue();

        if (isSet()) {
            if (value instanceof SecretValue) {
                return ((SecretValue) value).getValue();
            } else {
                throw new CoreException("vapi.data.optional.getvalue.mismatch",
                                        "Secret",
                                        value.getType().toString());
            }
        } else if (throwOnUnset) {
            throw new CoreException("vapi.data.optional.getvalue.unset");
        }
        return orElse;
    }

    public char[] getSecret() {
        return getSecretInternal(true, new char[] {});
    }

    public char[] getSecretOrElse(char[] orElse) {
        return getSecretInternal(false, orElse);
    }

    public char[] getSecretOrNull() {
        return getSecretOrElse(null);
    }

    public OptionalValue getOptional() {
        DataValue value = getValue();

        if (!isSet()) {
            throw new CoreException("vapi.data.optional.getvalue.unset");
        }

        if (value instanceof OptionalValue) {
            return (OptionalValue) value;
        }

        throw new CoreException("vapi.data.optional.getvalue.mismatch",
                                "Optional",
                                value.getType().toString());
    }

    public ListValue getList() {
        DataValue value = getValue();

        if (!isSet()) {
            throw new CoreException("vapi.data.optional.getvalue.unset");
        }

        if (value instanceof ListValue) {
            return (ListValue) value;
        }

        throw new CoreException("vapi.data.optional.getvalue.mismatch",
                                "List",
                                value.getType().toString());
    }

    public StructValue getStruct() {
        DataValue value = getValue();

        if (!isSet()) {
            throw new CoreException("vapi.data.optional.getvalue.unset");
        }

        if (value instanceof StructValue) {
            return (StructValue) value;
        }

        throw new CoreException("vapi.data.optional.getvalue.mismatch",
                                "Struct",
                                value.getType().toString());
    }

    @Override
    public void accept(ValueVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public OptionalValue copy() {
        return (value == null)? this : new OptionalValue(value.copy());
    }

    /**
     * Preserve the single {@link UNSET} instance when creating instances via
     * <em>Java</em> deserialization.
     * @see java.io.Serializable
     *
     * @return the {@link #UNSET} instance of this class
     */
    private Object readResolve() {
        return this.value == null ? UNSET : this;
    }
}
