/* **********************************************************
 * Copyright 2012-2014, 2016 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.data;

import static com.vmware.vapi.MessageFactory.getMessage;

import java.util.Arrays;
import com.vmware.vapi.CoreException;

/**
 * Value of type secret, which is intended to represent sensitive
 * information, like passwords.
 *
 * <p>The content of the value is stored in a {@code char[]} which allows
 * for cleaning the value from memory when it is no longer needed, this can
 * be done via {@link #clear()}.
 *
 * <p>In addition the actual content will not be returned by the {@link #toString()}
 * as a precaution for avoiding accidental displaying or logging it.
 */
public final class SecretValue implements DataValue {
    /**
     * Constant returned by the {@link #toString} method of {@link SecretValue}.
     */
    public static final String STRING_REPRESENTATION = "<secret>";

    private static final long serialVersionUID = 1L;

    /** The actual char array content of this SecretValue. */
    private final char[] value;

    /**
     * Create a value with the specified char array content.
     *
     * @param value char[]
     * @throws CoreException if the passed char array is null
     */
    public SecretValue(char[] value) {
        if (value == null) {
            throw new CoreException(getMessage("vapi.data.secret.null.chararray"));
        }
        this.value = value;
    }

    @Override
    public DataType getType() {
        return DataType.SECRET;
    }

    /**
     * Returns the underlying char array content. Will never be null.
     *
     * @return content
     */
    public char[] getValue() {
        return this.value;
    }

    /**
     * Clear the content inside SecretValue
     */
    public void clear() {
        Arrays.fill(this.value, ' ');
    }

    @Override
    public boolean equals(Object o) {
        if (o instanceof SecretValue) {
            SecretValue secret = (SecretValue) o;
            return Arrays.equals(this.value, secret.value);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(value);
    }

    /**
     * Always returns "&lt;secret&gt;" - Safety precaution for the case when
     * someone uses the toString directly on {@link SecretValue} object or on a
     * {@link StructValue} that contains a secret and for instance logs the
     * result.
     */
    @Override
    public String toString() {
        return STRING_REPRESENTATION;
    }

    @Override
    public void accept(ValueVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public SecretValue copy() {
        return new SecretValue(Arrays.copyOf(this.value, this.value.length));
    }
}
