/* **********************************************************
 * Copyright 2011-2015, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.data;

import static com.vmware.vapi.MessageFactory.getMessage;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.Message;
import com.vmware.vapi.internal.util.Validate;

public class StructDefinition extends DataDefinition {

    static Logger logger = LoggerFactory.getLogger(StructDefinition.class);

    private final Map<String, DataDefinition> fields;
    private final String name;

    /**
     * Constructor.
     *
     * @param name name for the structure; must not be <code>null</code>
     * @param fields field names and definitions for the structure;
     *               must not be <code>null</code>
     * @throws IllegalArgumentException if some of the arguments or the
     *         values in <code>fields</code> is <code>null</code>
     */
    public StructDefinition(String name, Map<String, DataDefinition> fields) {
        Validate.notNull(name);
        Validate.notNull(fields);

        this.name = name;
        // verify there are no null field definitions
        for (Map.Entry<String, DataDefinition> field : fields.entrySet()) {
            if (field.getValue() == null) {
                throw new IllegalArgumentException("Missing definition for field "
                    + field.getKey());
            }
        }
        this.fields = new HashMap<>(fields);
    }

    @Override
    public DataType getType() {
        return DataType.STRUCTURE;
    }

    public String getName() {
        return name;
    }

    public Set<String> getFieldNames() {
        return fields.keySet();
    }

    public DataDefinition getField(String field) {
        return fields.get(field);
    }

    public boolean hasField(String field) {
        return fields.containsKey(field);
    }

    public StructValue newInstance() {
        return new StructValue(getName());
    }


    /**
     * {@inheritDoc}
     *
     * <p>In addition, validates that <code>value</code>'s name matches
     * <code>this</code> definition name and <code>value</code>'s fields
     * are valid for <code>this</code> definition fields.
     */
    @Override
    public List<Message> validate(DataValue value) {
        List<Message> errors = super.validate(value);

        if (!errors.isEmpty()) {
            return errors;
        }

        // this is safe because super.validate() verified the type
        StructValue str = (StructValue) value;

        if (!getName().equals(str.getName())) {
            errors.add(getMessage("vapi.data.structure.name.mismatch",
                                  getName(),
                                  str.getName()));
            return errors;
        }

        /*
         * check to make sure no fields are missing and those that are present
         * are valid
         */
        for (String fieldName : fields.keySet()) {
            if (!str.hasField(fieldName)) {
                errors.add(getMessage("vapi.data.structure.field.missing",
                                      getName(), fieldName));
            } else {
                List<Message> subErrors;
                DataDefinition fieldDef = fields.get(fieldName);
                try {
                    DataValue fieldVal = str.getField(fieldName);
                    subErrors = fieldDef.validate(fieldVal);
                } catch (CoreException ex) {
                    subErrors = new LinkedList<>();
                    subErrors.addAll(ex.getExceptionMessages());
                }

                if (!subErrors.isEmpty()) {
                    errors.add(getMessage("vapi.data.structure.field.invalid",
                                          fieldName));
                    errors.addAll(subErrors);
                }
            }

            if (!errors.isEmpty()) {
                // stop processing at first detected error
                break;
            }
        }

        /*
         * Note the StructValue being validated can have additional fields
         * not known to this StructDefinition. It is required to tolerate them
         * because they might be coming from a client which has newer version of
         * the API definitions and there are extra fields added in the structure
         * for this newer version.
         */

        return errors;
    }

    @Override
    public void completeValue(DataValue value) {
        if (value != null && value.getType() == getType()) { // STRUCTURE or ERROR
            StructValue str = (StructValue) value;

            for (String fieldName : fields.keySet()) {
                if (!str.hasField(fieldName)) {
                    DataDefinition fieldDef = fields.get(fieldName);

                    if (fieldDef.getType() == DataType.OPTIONAL) {
                        // we can fill in the optionals that are missing
                        OptionalDefinition optDef =
                                (OptionalDefinition) fieldDef;

                        try {
                            str.setField(fieldName, optDef.newInstance());
                        } catch (CoreException ex) {
                            logger.debug("Unable to fill in optional value for "
                                + fieldName, ex);
                        }
                    }
                } else {
                    DataDefinition subElement = fields.get(fieldName);
                    subElement.completeValue(str.getField(fieldName));
                }
            }
        }
    }

    @Override
    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }

        if (!(other instanceof StructDefinition)) {
            return false;
        }
        StructDefinition otherStruct = (StructDefinition) other;

        return this.name.equals(otherStruct.name) &&
                this.fields.equals(otherStruct.fields);
    }

    @Override
    public int hashCode() {
        int hash = 17;
        hash += 31 * name.hashCode();
        hash += 31 * fields.hashCode();
        return hash;
    }

    @Override
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append(getName() + " {");
        for (Map.Entry<String, DataDefinition> e : fields.entrySet()) {
            buf.append("'");
            buf.append(e.getKey());
            buf.append("' : ");
            buf.append(e.getValue());
            buf.append(", ");
        }
        buf.append("}");
        return buf.toString();
    }

    @Override
    public void accept(DefinitionVisitor visitor) {
        visitor.visit(this);
    }
}
