/* **********************************************************
 * Copyright 2012-2013, 2018-2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.dsig.json;

import java.security.cert.X509Certificate;
import java.util.Arrays;

import com.vmware.vapi.internal.dsig.json.KeyStoreHelper;
import com.vmware.vapi.internal.util.Validate;

/**
 * This class is capable of retrieving the STS certificate trust chain from
 * different sources. This chain is needed to verify that the SAML token from
 * the signature is trusted.
 */
public final class DefaultStsTrustChain implements StsTrustChain {

    private final X509Certificate[] trustedChain;

    /**
     * Constructor.
     *
     * @param trustedChain must not be {@code null}, empty or have {@code null} elements
     */
    private DefaultStsTrustChain(X509Certificate[] trustedChain) {
        Validate.notEmpty(trustedChain);
        Validate.noNullElements(trustedChain);

        this.trustedChain = trustedChain;
    }

    public DefaultStsTrustChain() {
        trustedChain = null;
    }

    /**
     * @return the STS trust chain
     */
    public X509Certificate[] getStsTrustChain() {
        return Arrays.copyOf(trustedChain, trustedChain.length);
    }

    /**
     * Loads the STS trust chain from keystore
     *
     * @param fileName
     *            cannot be null.
     * @param password
     *            can be null.
     * @param certAlias
     *            cannot be null.
     * @return trust chain
     */
    public static DefaultStsTrustChain loadFromKeystore(String fileName,
            String password, String certAlias) {
        Validate.notNull(fileName);
        Validate.notNull(certAlias);

        KeyStoreHelper keyStore = new KeyStoreHelper(DefaultStsTrustChain.class
                .getResource(fileName).getFile(),
                (password == null) ? new char[] {} : password.toCharArray());
        // TODO add support for chain
        return new DefaultStsTrustChain(
                new X509Certificate[] { keyStore.getCertificate(certAlias) });
    }

    /**
     * Creates a {@link DefaultStsTrustChain} instance with predefined trusted
     * certificate chain.
     *
     * @param trustedChain must not be {@code null}, empty or have {@code null} elements
     * @return trust chain
     */
    public static DefaultStsTrustChain createInstance(X509Certificate[] trustedChain) {
        Validate.notNull(trustedChain);
        return new DefaultStsTrustChain(Arrays.copyOf(trustedChain, trustedChain.length));
    }
}
