/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc. All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.dsig.json;

import static com.vmware.vapi.core.ExecutionContext.SecurityContext.AUTHENTICATION_SCHEME_ID;

import java.io.UnsupportedEncodingException;
import java.util.Map;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.internal.protocol.common.json.JsonSecurityContextSerializer;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.RequestProcessor;

/**
 * This is an abstract processor that inserts the appropriate security context
 * data into the request. Is is an extension point for third party authentication
 * schemes.
 */
public abstract class SecurityContextProcessor implements RequestProcessor {

    public static final String SCHEME_ID_KEY = "schemeId";

    private static final Message DECODE_ERROR =
            MessageFactory.getMessage("vapi.proc.secctx.decoderequest");

    // TODO serializer should be injected and the processor will become protocol
    // agnostic
    private final JsonSecurityContextSerializer serializer =
            new JsonSecurityContextSerializer();

    @Override
    public byte[] process(byte[] request,
                          Map<String, Object> metadata,
                          Request vapiRequest) {
        Validate.notNull(request);
        Validate.notNull(metadata);

        // TODO security context could be obtained directly from the vapiRequest
        Object secCtx = metadata.get(RequestProcessor.SECURITY_CONTEXT_KEY);
        if (secCtx == null || !(secCtx instanceof SecurityContext)) {
            return request;
        }

        SecurityContext context = (SecurityContext) secCtx;
        Object schemeId = context.getProperty(AUTHENTICATION_SCHEME_ID);
        if (schemeId == null || !(schemeId instanceof String)) {
            return request;
        }
        byte[] result = request;
        if (isSchemeSupported((String)schemeId)) {
            try {
                String requestStr = new String(request, UTF8_CHARSET);
                Map<String, Object> props = getSecurityContextProperties(context);
                result = serializer.serializeSecurityContext(
                        props,requestStr).getBytes(UTF8_CHARSET);
            } catch (UnsupportedEncodingException e) {
                throw new CoreException(DECODE_ERROR);
            }
        }

        return result;
    }

    /**
     * @param requestedScheme the authentication scheme that is used with the
     *                        current security context
     * @return true if the processor is able to handle the requested scheme,
     *         false otherwise
     */
    public abstract boolean isSchemeSupported(String requestedScheme);

    /**
     * @param ctx the current security context. cannot be null.
     * @return map of the data that should be serialized as part of the security
     *         context. The security context is permitted to contain only Maps,
     *         Arrays and Strings. cannot be null.
     */
    public abstract Map<String, Object> getSecurityContextProperties(SecurityContext ctx);
}
