/*
 * Copyright (c) 2018, 2020 VMware, Inc. All rights reserved. -- VMware Confidential
 */
package com.vmware.vapi.internal.bindings;

import java.util.concurrent.CompletableFuture;

import com.vmware.vapi.bindings.client.AsyncCallback;
import com.vmware.vapi.bindings.Task;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.std.Progress;

/**
 * Factory class to wrap {@link CompletableFuture} in an {@link AsyncCallback}
 *
 * @param <T> type of the operation result; for <code>void</code> type, use
 *        <code>java.lang.Void</code>
 */
public class AsyncCallbackFactory<T> {

    /**
     * Wraps the provided {@link java.util.concurrent.CompletableFuture} into an
     * AsyncCallback.
     *
     * @param future which completes normally when async callback finishes
     *        properly or exceptionally when async callback throws an exception.
     *        Must not be {@code null}.
     * @param <T> the type of the operation result
     * @return callback wrapper of the provided future
     */
    public static <T> AsyncCallback<T> getCallback(CompletableFuture<T> future) {
        return new AsyncCallback<T>() {

            @Override
            public void onProgress(Progress progress) {
            }

            @Override
            public void onResult(T result) {
                future.complete(result);
            }

            @Override
            public void onError(RuntimeException error) {
                future.completeExceptionally(error);
            }
        };
    }

    /**
     * Wraps the provided {@link java.util.concurrent.CompletableFuture} into an
     * AsyncCallback.
     *
     * @param future which completes normally when async callback finishes
     *        properly or exceptionally when async callback throws an exception.
     *        Normal completion returns a {@link Task} object which contains
     *        taskId and the return type of the operation.
     *        Must not be {@code null}.
     * @param <T> the type of the operation result
     * @return callback wrapper of the provided future
     */
    public static <T> AsyncCallback<String> getTaskCallback(CompletableFuture<Task<T>> future, Type outputType) {
        return new AsyncCallback<String>() {

            @Override
            public void onProgress(Progress progress) {
            }

            @Override
            public void onResult(String result) {
                Task<T> taskResult = new Task<>(result, outputType);
                future.complete(taskResult);
            }

            @Override
            public void onError(RuntimeException error) {
                future.completeExceptionally(error);
            }
        };
    }
}
