/* **********************************************************
 * Copyright (c) 2013-2015, 2019, 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.bindings.type.TypeReference;
import com.vmware.vapi.data.ConstraintValidationException;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.data.ConstraintValidator;
import com.vmware.vapi.internal.data.TypeConverterDrivenConstraintValidator;
import com.vmware.vapi.internal.util.Validate;

/**
 * {@link ConstraintValidator} for {@literal @HasFieldsOf} constraint.
 */
public class HasFieldsOfValidator implements TypeConverterDrivenConstraintValidator {
    static final String CAPABILITY = "capability";
    static final String AT_CLASS = "@class";
    private static final String COMPUTE_POLICY_CAPABILITIES_PACKAGE = "com.vmware.vcenter.compute.policies";
    private static final String CREATE_SPEC = "create_spec";

    private static final Message FAILED_VALIDATION_MSG =
            MessageFactory.getMessage("vapi.data.structure.hasfieldsof.invalid");
    private final TypeReference<? extends StructType> structTypeRef;
    private TypeConverter converter;

    /**
     * Constructor.
     *
     * @param structTypeRef type reference for <code>StructType<code> or
     *                      descendant; describes the {@literal @HasFieldsOf}
     *                      constraint i.e. the structure type to which the
     *                      type conversion should always succeed; must not be
     *                      <code>null</code>
     */
    public HasFieldsOfValidator(
            TypeReference<? extends StructType> structTypeRef) {
        Validate.notNull(structTypeRef);
        this.structTypeRef = structTypeRef;
    }

    /**
     * This method returns a copy of the provided {@code dataValue} with the "capability"
     * {@code field} set if the dataValue:
     * <ul>
     *  <li>is an instance of {@link StructValue}</li>
     *  <li>is a compute policy "CreateSpec" {@code structure}</li>
     *  <li>does not have a populated "capability" {@code field}</li>
     * </ul>
     * <p>
     *
     * @param dataValue on which the "capability" field will be set
     * @return the provided {@code dataValue} if no mutation has been done or a deep copy of the
     *         {@code dataValue} with "capability" field set to "doesn't matter"
     */
    // TODO: Delete this hack once compute policy confirm that Horizon View no longer needs it.
    static DataValue populateCapabilityFieldForComputePolicy(DataValue dataValue) {
        if (!(dataValue instanceof StructValue)) {
            return dataValue;
        }

        StructValue asStructValue = (StructValue) dataValue;
        String name = asStructValue.getName();
        if (name == null || name.isEmpty()) {
            if (asStructValue.hasField(AT_CLASS)) {
                try {
                    name = asStructValue.getString(AT_CLASS);
                } catch (CoreException ex) {
                    return dataValue;
                }
            } else {
                name = null;
            }
        }

        if (name != null &&
                name.startsWith(COMPUTE_POLICY_CAPABILITIES_PACKAGE) &&
                name.endsWith(CREATE_SPEC)) {
            if (asStructValue.hasField(CAPABILITY)) {
                return dataValue;
            }
            StructValue copy = asStructValue.copy();
            // we set the capability field just so convertTo validation passes in case
            // it is missing but is required. For the validation its value doesn't matter.
            copy.setField(CAPABILITY, "doesn't matter");
            return copy;
        }

        return dataValue;
    }

    @Override
    public void validate(DataValue dataValue) {
        if (converter == null) {
            converter = new TypeConverterImpl();
        }

        try {
            dataValue = populateCapabilityFieldForComputePolicy(dataValue);
            converter.convertToJava(dataValue, structTypeRef.resolve());
        } catch (ConverterException e) {
            throw new ConstraintValidationException(FAILED_VALIDATION_MSG, e);
        }
    }

    @Override
    public void validate(DataValue dataValue, TypeConverter converter) {
        Validate.notNull(converter);
        dataValue = populateCapabilityFieldForComputePolicy(dataValue);
        converter.convertToJava(dataValue, structTypeRef.resolve());
    }
}
