/* **********************************************************
 * Copyright 2014, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings;

import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.IntegerValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.client.exception.BindingsException;
import com.vmware.vapi.std.Progress;

/**
 * Converter of progress notifications. Converts between {@link Progress} and
 * {@link DataValue}.
 */
public final class ProgressConverter {

    public static final String PROGRESS_STRUCT_NAME =
            "com.vmware.vapi.std.progress";

    public static final String PROGRESS_FIELD_MIN = "min";

    public static final String PROGRESS_FIELD_MAX = "max";

    public static final String PROGRESS_FIELD_CURRENT = "current";

    /**
     * Creates a {@link DataValue} from a progress binding.
     *
     * @param progress progress binding; must not be <code>null</code>
     * @return progress value
     */
    public static StructValue toValue(Progress progress) {
        Validate.notNull(progress);
        StructValue s = new StructValue(PROGRESS_STRUCT_NAME);
        s.setField(PROGRESS_FIELD_MIN, progress.getMin());
        s.setField(PROGRESS_FIELD_MAX, progress.getMax());
        s.setField(PROGRESS_FIELD_CURRENT, progress.getCurrent());
        return s;
    }

    /**
     * Creates a progress binding from a {@link DataValue}.
     *
     * @param value progress value; must not be <code>null</code>
     * @return progress binding
     * @throws BindingsException if the progress representation is
     * invalid
     */
    public static Progress fromValue(DataValue value) {
        Validate.notNull(value);
        if (!(value instanceof StructValue)) {
            throw new BindingsException(
                    String.format(
                            "Invalid progress notification format: expected structure '%s', found '%s' value",
                            PROGRESS_STRUCT_NAME, value.getType().name()));
        }
        StructValue s = (StructValue) value;
        if (!PROGRESS_STRUCT_NAME.equals(s.getName())) {
            throw new BindingsException(
                    String.format(
                            "Invalid progress notification format: expected structure '%s', found structure '%s'",
                            PROGRESS_STRUCT_NAME, s.getName()));
        }
        long min = getProgressField(s, PROGRESS_FIELD_MIN);
        long max = getProgressField(s, PROGRESS_FIELD_MAX);
        long current = getProgressField(s, PROGRESS_FIELD_CURRENT);
        try {
            return new Progress(min, max, current);
        } catch (IllegalArgumentException ex) {
            // violation of the progress invariant
            throw new BindingsException("Invalid progress notification: "
                    + ex.getMessage(), ex);
        }
    }

    private static long getProgressField(StructValue progressStruct,
                                         String fieldName) {
        if (!progressStruct.hasField(fieldName)) {
            throw new BindingsException(
                    String.format(
                            "Invalid progress notification format: missing field '%s'",
                            fieldName));
        }
        DataValue fieldValue = progressStruct.getField(fieldName);
        if (!(fieldValue instanceof IntegerValue)) {
            throw new BindingsException(
                    String.format(
                            "Invalid progress notification format: expected integer field '%s', found '%s' value",
                            fieldName, fieldValue.getType().name()));
        }
        return ((IntegerValue) fieldValue).getValue();
    }

}
