/* **********************************************************
 * Copyright (c) 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings;

import java.util.Collection;

import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.Consumer;
import com.vmware.vapi.core.MethodResult;

/**
 * Streamlines the handling of an invocation result by converting the response
 * and invoking further either the {@link #onSuccess(Object, Consumer)} or
 * {@link #onFailure(RuntimeException)} methods depending on the outcome.
 *
 * @param <T> the return type of the invoked operation
 */
abstract class ResultTranslatingHandle<T> extends AsyncHandle<MethodResult> {
    protected final Stub stub;
    protected final Type outputType;
    protected final Collection<Type> errorTypes;

    ResultTranslatingHandle(Stub stub,
                            Type outputType,
                            Collection<Type> errorTypes) {
        this.stub = stub;
        this.outputType = outputType;
        this.errorTypes = errorTypes;
    }

    @Override
    public void setResult(MethodResult result) {
        if (result.equals(MethodResult.EMPTY)) {
            onSuccess(null, null);
        } else if (result.success()) {
            onSuccess(stub.<T> convert(result.getOutput(), outputType),
                      result.getNext());
        } else {
            onFailure(stub.convertError(result.getError(), errorTypes));
        }

        /*
         * TODO remove this and add subscribers for the "Mono" responses.
         */
        postProcessResponse(result);
    }

    /**
     * If we figure out that this is a "Mono" response (borrowed from project
     * reactor terminology), meaning a single response, then initiate a
     * resumeRead action. The read will go through the HTTP terminal frame,
     * hence, closing the underlying HTTP response.
     *
     * NOTE - when the response is not "Mono", rather a stream, the Publisher
     * overrides this behavior.
     */
    protected void postProcessResponse(MethodResult result) {
        Consumer<AsyncHandle<MethodResult>> nextHandleConsumer = result.getNext();
        if (nextHandleConsumer != null) {
            // <null> is supplied to forbid any further readings.
            nextHandleConsumer.accept(null);
        }
    }

    @Override
    public void setError(RuntimeException error) {
        onFailure(BindingsExceptionTranslator.translate(error));
    }

    /**
     * Invoked upon a successful response with the result converted to the
     * expected return type.
     *
     * @param result the successful response of the invocation
     * @param next the result of the {@link MethodResult#getNext()} invocation
     */
    abstract void onSuccess(T result, Consumer<AsyncHandle<MethodResult>> next);

    /**
     * Invoked upon a failed response with the error properly translated for
     * further consumption.
     *
     * @param error
     */
    abstract void onFailure(RuntimeException error);

}
