/* **********************************************************
 * Copyright (c) 2012-2020, 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.bindings.Structure;
import com.vmware.vapi.bindings.StubConfigurationBase;
import com.vmware.vapi.bindings.client.AsyncCallback;
import com.vmware.vapi.bindings.client.AsyncCallbackSyncAdapter;
import com.vmware.vapi.bindings.client.InvocationConfig;
import com.vmware.vapi.bindings.client.RetryPolicy;
import com.vmware.vapi.bindings.client.RetryPolicy.RetryContext;
import com.vmware.vapi.bindings.client.RetryPolicy.RetrySpec;
import com.vmware.vapi.bindings.type.AnyErrorType;
import com.vmware.vapi.bindings.type.ErrorType;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.bindings.type.TypeReference;
import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.Consumer;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.ApplicationData;
import com.vmware.vapi.core.ExecutionContext.RuntimeData;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.core.InterfaceIdentifier;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.diagnostics.LogDiagnosticUtil;
import com.vmware.vapi.diagnostics.LogDiagnosticsConfigurator;
import com.vmware.vapi.diagnostics.Slf4jMDCLogConfigurator;
import com.vmware.vapi.internal.bindings.convert.NameToTypeResolver;
import com.vmware.vapi.internal.bindings.convert.UniTypeConverter;
import com.vmware.vapi.internal.bindings.convert.impl.DefaultConverterFactory;
import com.vmware.vapi.internal.bindings.convert.impl.JavaClassStructConverter;
import com.vmware.vapi.internal.bindings.convert.impl.MapBasedNameToTypeResolver;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.std.Progress;
import com.vmware.vapi.std.errors.Error;

/**
 * Base class for all generated stubs for vAPI services.
 */
public class Stub {
    private static final Logger LOGGER = LoggerFactory.getLogger(Stub.class);

    /** identifier of the vAPI service represented by this stub  */
    protected final InterfaceIdentifier ifaceId;

    /** API provider for the protocol connection */
    protected final ApiProvider apiProvider;

    /** converter between API runtime types and bindings types */
    protected final TypeConverter converter;

    protected final SecurityContext securityContext;

    protected final RetryPolicy retryPolicy;

    /**
     * Constructor.
     *
     * <p>Uses {@code null} for {@TypeConverter}.
     *
     * @see #Stub(ApiProvider, TypeConverter, String, StubConfigurationBase)
     */
    protected Stub(ApiProvider apiProvider,
                   InterfaceIdentifier ifaceId,
                   StubConfigurationBase config) {
        this(apiProvider, null, ifaceId, config);
    }

    /**
     * Constructor.
     *
     * @param apiProvider client side {@code ApiProvider} to handle actual
     *        remote invocation
     * @param typeConverter
     * @param ifaceId identifier of the service represented by this stub
     * @param config configuration for this stub
     */
    protected Stub(ApiProvider apiProvider,
                   TypeConverter typeConverter,
                   InterfaceIdentifier ifaceId,
                   StubConfigurationBase config) {
        Validate.notNull(apiProvider);
        Validate.notNull(config);
        Validate.notNull(ifaceId);

        this.ifaceId = ifaceId;
        this.apiProvider = apiProvider;
        this.converter =
                typeConverter != null ? typeConverter
                                      : createTypeConverter(config.getErrorTypes());
        securityContext = config.getSecurityContext();
        retryPolicy = config.getRetryPolicy();
    }

    /**
     * Method to help invoke method on <code>ApiProvider</code>.
     *
     * <p>Designed to be invoked by bindings stub generated code.
     *
     * @param methodId          identifier of the method to be invoked
     * @param structBuilder     <code>StructBuilder</code> for the method input
     *                          structure
     * @param inputType         type of method input
     * @param outputType        type of method output
     * @param errorTypes        types for expected errors for the method
     * @param invocationConfig  configuration for the method invocation
     * @return  result from method invocation converted to Java native type
     *
     * @throws CoreException   if authentication is required but no
     *                         security configuration is available
     * @throws Error
     */
    protected <T> T invokeMethod(MethodIdentifier methodId,
                                 StructValueBuilder structBuilder,
                                 StructType inputType,
                                 Type outputType,
                                 Collection<Type> errorTypes,
                                 InvocationConfig invocationConfig) {
        AsyncCallbackSyncAdapter<T> future = new AsyncCallbackSyncAdapter<>();
        invokeMethodAsync(methodId,
                          structBuilder,
                          inputType,
                          outputType,
                          errorTypes,
                          invocationConfig,
                          future);
        try {
            return future.get();
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            // TODO: use CoreException
            throw new RuntimeException(ex);
        }
    }

    protected <T> void invokeMethodAsync(MethodIdentifier methodId,
                                         StructValueBuilder structBuilder,
                                         StructType inputType,
                                         Type outputType,
                                         Collection<Type> errorTypes,
                                         InvocationConfig invocationConfig,
                                         final AsyncCallback<T> asyncCallback) {
        StructValue inputValue;
        try {
            inputValue = structBuilder.getStructValue();
            validateInput(inputType, inputValue, methodId);
        } catch (RuntimeException e) {
            asyncCallback.onError(BindingsExceptionTranslator.translate(e));
            return;
        }
        ExecutionContext execCtx = getExecutionContext(invocationConfig);
        ResultTranslatingHandle<T> handle = new ResultTranslatingHandle<T>(this,
                                                                           outputType,
                                                                           errorTypes) {
            @Override
            public void updateProgress(DataValue progress) {
                asyncCallback.onProgress(convertProgress(progress));
            }

            @Override
            public void onSuccess(T result,
                                  Consumer<AsyncHandle<MethodResult>> next) {
                asyncCallback.onResult(result);
            }

            @Override
            public void onFailure(RuntimeException error) {
                asyncCallback.onError(error);
            }
        };

        final String serviceId = methodId.getInterfaceIdentifier().getName();
        final String operationId = methodId.getName();
        invoke(handle,
               serviceId,
               operationId,
               inputValue,
               outputType,
               errorTypes,
               execCtx,
               asyncCallback,
               0);
    }

    private <T> void invoke(final ResultTranslatingHandle<T> handle,
                            final String serviceId,
                            final String operationId,
                            final StructValue inputValue,
                            final Type outputType,
                            final Collection<Type> errorTypes,
                            final ExecutionContext executionContext,
                            final AsyncCallback<T> asyncCallback,
                            final int invocationAttempt) {
        AsyncHandle<MethodResult> ah = handle;
        if (isRetryingConfigured()) {
            ah = new RetryingHandle<T>(this,
                                       outputType,
                                       errorTypes,
                                       handle,
                                       serviceId,
                                       operationId,
                                       executionContext,
                                       inputValue,
                                       invocationAttempt) {
                @Override
                void onRetry(ExecutionContext retryContext) {
                    invoke(handle,
                           serviceId,
                           operationId,
                           inputValue,
                           outputType,
                           errorTypes,
                           retryContext,
                           asyncCallback,
                           invocationAttempt + 1);
                }
            };
        }

        LogDiagnosticsConfigurator logConfig = new Slf4jMDCLogConfigurator();
        try {
            logConfig.configureContext(LogDiagnosticUtil
                    .getDiagnosticContext(executionContext));
            apiProvider.invoke(serviceId,
                               operationId,
                               inputValue,
                               executionContext,
                               ah);
        } finally {
            logConfig.cleanUpContext(LogDiagnosticUtil.getDiagnosticKeys());
        }
    }

    /**
     * Validates the constraints defined over the input structure.
     */
    void validateInput(StructType inputType,
                       StructValue inputValue,
                       MethodIdentifier methodId) {
        ValidatorUtil.validate(inputType, inputValue, methodId);
    }

    /**
     * Method to help invoke an {@code @Stream} method on {@code ApiProvider}.
     *
     * <p>
     * Designed to be invoked by bindings stub generated code. A conversion from
     * {@code Object} to {@code Publisher<T>} is done by the generated code.
     *
     * @param methodId identifier of the method to be invoked
     * @param structBuilder <code>StructValueBuilder</code> for the method input
     *        structure
     * @param inputType type of method input
     * @param outputType type of method output
     * @param errorTypes types for expected errors for the method
     * @param invocationConfig configuration for the method invocation
     * @return result from method invocation converted to Java native type
     *
     * @throws CoreException if authentication is required but no security
     *         configuration is available
     * @throws Error
     */
    protected <T> Object invokeStreamMethod(MethodIdentifier methodId,
                                            StructValueBuilder structBuilder,
                                            StructType inputType,
                                            Type outputType,
                                            Collection<Type> errorTypes,
                                            InvocationConfig invocationConfig) {
        StructValue inputValue;
        try {
            inputValue = structBuilder.getStructValue();
            validateInput(inputType, inputValue, methodId);
        } catch (RuntimeException e) {
            throw BindingsExceptionTranslator.translate(e);
        }

        return new StreamPublisher<T>(this,
                                      methodId,
                                      inputValue,
                                      inputType,
                                      outputType,
                                      errorTypes,
                                      invocationConfig);
    }

    /**
     * Looks for <code>ErrorType</code> in <code>types</code> that matches
     * the <code>errValue</errValue> name.
     *
     * @param errValue the <code>ErrorValue</code> to resolve
     * @param errorTypes the error types declared by the method for the
     *        invocation being processed
     *
     * @return <code>ErrorType</code> if error is resolved; or <code>null</code>
     *         otherwise
     */
    ErrorType resolveErrorType(ErrorValue errValue,
                               Collection<Type> errorTypes) {
        if (errValue == null) {
            return null;
        }

        String errorValueName = errValue.getName();
        if (errorTypes != null) {
            // try resolving using ErrorTypes from IDL (throws clause)
            // this can handle custom errors
            for (Type t : errorTypes) {
                ErrorType errType = (ErrorType) ((TypeReference<?>) t).resolve();
                if (errorValueName.equals(errType.getName())) {
                    return errType;
                }
            }
        }

        // can not resolve the error name
        return null;
    }

    @Override
    public String toString() {
        return getClass().getName() + "<" + ifaceId.getName() + ">";
    }

    /**
     * Prepares an execution context for retrying of method invocation.
     *
     * @param oldContext execution context from the previous invocation
     *                   attempt; must not be <code>null</code>
     * @param retrySpec specification for the new invocation attempt; must not
     *                  be <code>null</code>
     * @return execution context
     */
    private static ExecutionContext getExecutionContextForRetry(
            ExecutionContext oldContext,
            RetrySpec retrySpec) {
        if (retrySpec.getInvocationConfig() == null) {
            return oldContext;
        }
        InvocationConfig invocationConfig = retrySpec.getInvocationConfig();
        ExecutionContext newContext = invocationConfig.getExecutionContext();
        if (newContext == null) {
            return oldContext;
        }
        ApplicationData newData = newContext.retrieveApplicationData();
        ApplicationData oldData = oldContext.retrieveApplicationData();
        if (newData == null) {
            newData = oldData;
        } else if (getOpId(newData) == null) {
            String opId = getOpId(oldData);
            if (opId != null) {
                // copy opId from the old context
                newData = ApplicationData.merge(newData,
                                                LogDiagnosticUtil.OPERATION_ID,
                                                opId);
            }
        }
        SecurityContext newSecurity = newContext.retrieveSecurityContext();
        if (newSecurity == null) {
            newSecurity = oldContext.retrieveSecurityContext();
        }
        return new ExecutionContext(newData,
                                    newSecurity,
                                    newContext.retrieveRuntimeData());
    }

    /**
     * Prepares an execution context for method invocation.
     *
     * @param invocationConfig configuration for the invocation
     * @return execution context
     */
    ExecutionContext getExecutionContext(InvocationConfig invocationConfig) {
        return new ExecutionContext(getApplicationData(invocationConfig),
                                    getSecurityContext(invocationConfig),
                                    getRuntimeData(invocationConfig));
    }

    private SecurityContext getSecurityContext(
            InvocationConfig invocationConfig) {
        if (invocationConfig != null
                && invocationConfig.getExecutionContext() != null
                && invocationConfig.getExecutionContext()
                        .retrieveSecurityContext() != null) {
            return invocationConfig.getExecutionContext()
                    .retrieveSecurityContext();
        } else {
            return this.securityContext;
        }
    }

    private static ApplicationData getApplicationData(InvocationConfig ic) {
        if (ic != null) {
            ExecutionContext ec = ic.getExecutionContext();
            if (ec != null) {
                ApplicationData ad = ec.retrieveApplicationData();
                if (ad != null) {
                    if (ad.getAllProperties().containsKey(
                                              LogDiagnosticUtil.OPERATION_ID)) {
                        return ad;
                    }
                    return ApplicationData.merge(ad,
                                                 LogDiagnosticUtil.OPERATION_ID,
                                                 generateNewOpId());
                }
            }
        }

        return new ApplicationData(LogDiagnosticUtil.OPERATION_ID, generateNewOpId());
    }

    private static String generateNewOpId() {
        return UUID.randomUUID().toString();
    }

    private RuntimeData getRuntimeData(InvocationConfig invocationConfig) {
        if (invocationConfig == null) {
            return null;
        }

        ExecutionContext executionContext = invocationConfig.getExecutionContext();
        if (executionContext == null) {
            return null;
        }

        return executionContext.retrieveRuntimeData();
    }

    private static String getOpId(ApplicationData data) {
        if (data == null) {
            return null;
        }
        return data.getProperty(LogDiagnosticUtil.OPERATION_ID);
    }

    <T> T convert(DataValue value, Type type) {
        try {
            return converter.convertToJava(value, type);
        } catch (RuntimeException e) {
            throw BindingsExceptionTranslator.translate(e);
        }
    }

    private Progress convertProgress(DataValue progressValue) {
        return ProgressConverter.fromValue(progressValue);
    }

    RuntimeException convertError(ErrorValue error,
            Collection<Type> errorTypes) {
        Type type = resolveErrorType(error, errorTypes);
        if (type == null) {
            // unknown error type, let the type converter handle it
            type = new AnyErrorType();
        }
        return convert(error, type);
    }

    private static TypeConverter createTypeConverter(Set<ErrorType> typeSet) {
        Map<String, StructType> errorTypes =
                MapBasedNameToTypeResolver.augmentWithStandardErrors(typeSet);
        NameToTypeResolver errorsResolver = new MapBasedNameToTypeResolver(errorTypes);
        return new TypeConverterImpl(new StubConverterFactory(errorsResolver));
    }

    /**
     * Client side (stub) converter factory which skips unset optional fields when
     * converting native types to {@code StructValue}, but not for {@code ErrorValue}.
     *
     * <p>Skipping a field means that it is not included at all in the parent structure.
     *
     * <p>In all other aspects this factory is identical to {@link DefaultConverterFactory}.
     */
    static final class StubConverterFactory extends DefaultConverterFactory {
        // true - skip unset optional fields converting to DataValue
        private final UniTypeConverter<StructValue, StructType> skipUnsetStructConverter =
                new JavaClassStructConverter<>(StructValue.class, Structure.class, true);

        StubConverterFactory(NameToTypeResolver nameToTypeResolver) {
            super(nameToTypeResolver);
        }

        @Override
        public UniTypeConverter<StructValue, StructType> getStructConverter() {
            return skipUnsetStructConverter;
        }

        /*
         * Note: the purpose of this factory is to remove the unset optional
         * fields from the wire when sending request from static language
         * bindings. This is a workaround for a bug which was shipped in
         * vsphere 6.0 because of which the Java vAPI server is rejecting
         * requests with unknown fields, even if they are unset optional (it
         * must accept them as a versioning requirement). Thus new version
         * SDKs (e.g. vsphere 6.1) can't invoke the old server APIs (vsphere 6.0)
         * for API operations that has additional parameter (or structure field)
         * added in the new version of the API definition.
         * An example of such problem (reported internally during 6.1 development)
         * is http://bugzilla.eng.vmware.com/show_bug.cgi?id=1534445
         *
         * The actual bug on the Java server side was fixed with
         * https://bugzilla.eng.vmware.com/show_bug.cgi?id=1364821#c7
         * https://reviewboard.eng.vmware.com/r/725631/
         *
         * This fix will get into vsphere 6.1 and in addition I (mcvetanov) am
         * pushing it for backport for vsphere 6.0 Update 2 release.
         *
         * Note: the factory use here doesn't skip the unset optional fields
         * for ErrorValue, because of a bug in ErrorDefinition .complete
         * which was shipped with vSphere 6.0 and prevent proper completion
         * of ErrorValues, thus breaking the versioning requirement for
         * vAPI server to handle old client requests (with potentially missing
         * fields).
         *
         * This bug was fixed with http://p4web.eng.vmware.com/3886580?ac=10.
         */
    }

    /**
     * @return the context to use for retrying an invocation; {@code null} if
     *         there is no need to retry
     */
    ExecutionContext getRetryContext(String serviceId,
                                       String operationId,
                                       ExecutionContext executionContext,
                                       StructValue inputValue,
                                       RuntimeException error,
                                       int invocationAttempt) {
        RetryContext context = new RetryContext(serviceId,
                                                operationId,
                                                executionContext,
                                                inputValue);
        RetrySpec retrySpec = null;
        try {
            retrySpec = retryPolicy
                    .onInvocationError(error, context, invocationAttempt);
        } catch (RuntimeException e) {
            if (LOGGER.isWarnEnabled()) {
                String message = String.format("onInvocationError call on %s for %s.%s failed",
                                               retrySpec,
                                               serviceId,
                                               operationId);
                LOGGER.warn(message, e);
            }
        }

        if (retrySpec == null) {
            return null;
        }
        return getExecutionContextForRetry(executionContext, retrySpec);
    }

    boolean isRetryingConfigured() {
        return retryPolicy != null;
    }
}
