/* **********************************************************
 * Copyright (c) 2012-2022 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings;

import org.apache.commons.codec.binary.Base64;

import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.bindings.type.AnyErrorType;
import com.vmware.vapi.bindings.type.BinaryType;
import com.vmware.vapi.bindings.type.BooleanType;
import com.vmware.vapi.bindings.type.DateTimeType;
import com.vmware.vapi.bindings.type.DoubleType;
import com.vmware.vapi.bindings.type.DynamicStructType;
import com.vmware.vapi.bindings.type.EnumType;
import com.vmware.vapi.bindings.type.ErrorType;
import com.vmware.vapi.bindings.type.IdType;
import com.vmware.vapi.bindings.type.IntegerType;
import com.vmware.vapi.bindings.type.ListType;
import com.vmware.vapi.bindings.type.MapType;
import com.vmware.vapi.bindings.type.OpaqueType;
import com.vmware.vapi.bindings.type.OptionalType;
import com.vmware.vapi.bindings.type.SecretType;
import com.vmware.vapi.bindings.type.SetType;
import com.vmware.vapi.bindings.type.StringType;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.bindings.type.TypeReference;
import com.vmware.vapi.bindings.type.TypeVisitor;
import com.vmware.vapi.bindings.type.UriType;
import com.vmware.vapi.bindings.type.VoidType;
import com.vmware.vapi.data.BlobValue;
import com.vmware.vapi.data.BooleanValue;
import com.vmware.vapi.data.DataType;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.DoubleValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.data.IntegerValue;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.SecretValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.data.VoidValue;
import com.vmware.vapi.internal.bindings.convert.ConverterFactory;
import com.vmware.vapi.internal.bindings.convert.impl.ConvertUtil;
import com.vmware.vapi.internal.bindings.convert.impl.DefaultConverterFactory;
import com.vmware.vapi.internal.util.Validate;

/**
 * Converts values between Java data model and API runtime data model.
 */
public final class TypeConverterImpl implements TypeConverter {

    /**
     * Visitor which converts a vAPI data value to a Java object. This visitor
     * should be accepted by the (expected) type of the value which has to be
     * converted.
     */
    private class ValueToJavaVisitor implements TypeVisitor {

        private static final String UNRESOLVEABLE_NAME = "unresolved_39DFA178-BED8-4A21-BF2F-2F648C0DDFF4";

        /**
         * Type converter needed to feed single-type converters.
         */
        private final TypeConverter typeConverter;

        /**
         * Value which will be converted to a Java object.
         */
        private final DataValue value;

        /**
         * Output of the visitor.
         */
        private Object obj;

        /**
         * @param typeConverer type converter
         * @param value vAPI data value which will be converted to a Java object.
         */
        ValueToJavaVisitor(TypeConverter typeConverter, DataValue value) {
            this.typeConverter = typeConverter;
            this.value = value;
        }

        /**
         * The output of the visitor.
         *
         * @return Java object which corresponds to the data value that was
         *         specified when creating the visitor.
         */
        <T> T getJavaObject() {
            @SuppressWarnings("unchecked")
            T result = (T) obj;

            return result;
        }

        /**
         * Verify we are given a <code>VoidValue</code> and convert it to a Java
         * object.
         */
        @Override
        public void visit(VoidType type) {
            VoidValue v = ConvertUtil.narrowType(value, VoidValue.class);
            obj = converterFactory.getVoidConverter().fromValue(v);
        }

        /**
         * Verify we are given a <code>BooleanValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(BooleanType type) {
            BooleanValue v = convertDynamically(false)
                    ? BooleanValue.getInstance(Boolean.parseBoolean(getAsString(value)))
                    : ConvertUtil.narrowType(value, BooleanValue.class);
            obj = converterFactory.getBooleanConverter().fromValue(v);
        }

        /**
         * Verify we are given an <code>IntegerValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(IntegerType type) {
            IntegerValue v = convertDynamically(false)
                    ? new IntegerValue(Long.parseLong(getAsString(value)))
                    : ConvertUtil.narrowType(value, IntegerValue.class);
            obj = converterFactory.getIntegerConverter().fromValue(v);
        }

        /**
         * Verify we are given a <code>DoubleValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(DoubleType type) {
            DoubleValue v = convertDynamically(false)
                    ? new DoubleValue(Double.parseDouble(getAsString(value)))
                    : ConvertUtil.narrowType(value, DoubleValue.class);
            obj = converterFactory.getDoubleConverter().fromValue(v);
        }

        /**
         * Verify we are given a <code>StringValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(StringType type) {
            StringValue v = ConvertUtil.narrowType(value, StringValue.class);
            obj = converterFactory.getStringConverter().fromValue(v);
        }

        /**
         * Verify we are given a <code>BlobValue</code> and convert it to a Java
         * object.
         */
        @Override
        public void visit(BinaryType type) {
            BlobValue v = convertDynamically(false)
                    ? new BlobValue(Base64.decodeBase64(getAsString(value)))
                    : ConvertUtil.narrowType(value, BlobValue.class);
            obj = converterFactory.getBinaryConverter().fromValue(v);
        }

        /**
         * Verify we are given a <code>SecretValue</code> and convert it to a Java
         * object.
         */
        @Override
        public void visit(SecretType type) {
            SecretValue v = convertDynamically(false)
                    ? new SecretValue(getAsString(value).toCharArray())
                    : ConvertUtil.narrowType(value, SecretValue.class);
            obj = converterFactory.getSecretConverter().fromValue(v);
        }

        /**
         * Verify we are given a <code>StringValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(DateTimeType type) {
            StringValue v = ConvertUtil.narrowType(value, StringValue.class);
            obj = converterFactory.getDateTimeConverter().fromValue(v);
        }

        /**
         * Verify we are given a <code>StringValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(UriType type) {
            StringValue v = ConvertUtil.narrowType(value, StringValue.class);
            obj = converterFactory.getUriConverter().fromValue(v);
        }

        /**
         * Verify we are given a <code>StringValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(IdType idType) {
            StringValue v = ConvertUtil.narrowType(value, StringValue.class);
            obj = converterFactory.getIdConverter().fromValue(v);
        }

        /**
         * Verify we are given an <code>OptionalValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(OptionalType type) {
            OptionalValue v = null;
            if (value != null) {
                /*
                 * We tolerate the absence of OptionalValue wrappers. This
                 * handles 2 cases of missing OptionalValues - rest native
                 * (metadata-free JSON deserialization) - "Legacy" (MD-based)
                 * REST deserialization of DynamicStructure fields without type
                 * identifier on the wire
                 */
                boolean addOptional = value.getType() != DataType.OPTIONAL;
                v = addOptional ? new OptionalValue(value)
                                : ConvertUtil.narrowType(value, OptionalValue.class);
            }
            obj = converterFactory.getOptionalConverter().fromValue(
                    v, type, typeConverter);
        }

        /**
         * Verify we are given a <code>ListValue</code> and convert it to a Java
         * object.
         */
        @Override
        public void visit(ListType type) {
            ListValue v = convertListValue();
            obj = converterFactory.getListConverter().fromValue(
                    v, type, typeConverter);
        }

        @Override
        public void visit(SetType type) {
            ListValue v = convertListValue();
            obj = converterFactory.getSetConverter().fromValue(
                    v, type, typeConverter);
        }

        @Override
        public void visit(MapType type) {
            DataValue v;
            if (value.getType() == DataType.STRUCTURE) {
                v = value;
            } else {
                // TODO: this special handling could move down into the map
                //       converter, so we don't have to branch here
                v = convertListValue();
            }

            obj = converterFactory.getMapConverter().fromValue(
                        v, type, typeConverter);
        }

        /**
         * Verify we are given a <code>StructValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(StructType type) {
            StructValue structValue = convertDynamically(true)
                    ? new StructValue(type.getName())
                    : ConvertUtil.narrowType(value, StructValue.class);

            // validate all constraints defined over the structure
            type.validate(structValue);

            obj = converterFactory.getStructConverter().fromValue(
                    structValue, type, typeConverter);
        }

        private ListValue convertListValue() {
            return convertDynamically(true)
                    ? new ListValue()
                    : ConvertUtil.narrowType(value, ListValue.class);
        }

        /**
         * The output Java object is the same as the input vAPI data value.
         */
        @Override
        public void visit(OpaqueType type) {
            obj = converterFactory.getOpaqueConverter().fromValue(value);
        }

        /**
         * Resolve the type reference. Visit the actual type that the reference
         * points to.
         */
        @Override
        public void visit(TypeReference<? extends Type> type) {
            Type referredType = type.resolve();
            referredType.accept(this);
        }

        /**
         * Verify we are given a <code>StringValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(EnumType type) {
            StringValue v = ConvertUtil.narrowType(value, StringValue.class);
            obj = converterFactory.getEnumConverter().fromValue(v, type, typeConverter);
        }

        /**
         * Verify we are given an <code>ErrorValue</code> and convert it to a
         * Java object.
         */
        @Override
        public void visit(ErrorType type) {
            ErrorValue errorValue = coerceErrorValue(type.getName());

            // validate all constraints defined over the error
            type.validate(errorValue);

            obj = converterFactory.getErrorConverter().fromValue(
                    errorValue, type, typeConverter);
        }

        @Override
        public void visit(DynamicStructType type) {
            StructValue structValue =
                ConvertUtil.narrowType(value, StructValue.class);

            type.validate(structValue, reusableThis());
            obj = converterFactory.getDynamicStructureConverter().fromValue(
                    structValue, type, typeConverter);
        }

        @Override
        public void visit(AnyErrorType type) {
            ErrorValue errorValue = coerceErrorValue(UNRESOLVEABLE_NAME);
            obj = converterFactory.getAnyErrorConverter().fromValue(
                    errorValue, type, typeConverter);
        }

        /**
         * @param emptyValueAllowed true if the value field member is expected
         *                          to be empty StringValue
         * @return true if the conversion should be dynamic
         */
        private boolean convertDynamically(boolean emptyValueAllowed) {
            if (value.getType() == DataType.STRING) {
                StringValue stringValue = ConvertUtil.narrowType(value, StringValue.class);
                if (emptyValueAllowed) {
                    return stringValue.getValue().isEmpty();
                } else {
                    return !stringValue.getValue().isEmpty();
                }
            }

            return false;
        }

        /**
         * Coerces the input value to {@link ErrorValue}. Empty string and
         * {@link StructValue} are allowed. {@link StructValue} is converted to
         * {@link ErrorValue} with a specified type name.
         *
         * @param errName error type name to put in the {@link ErrorValue} as
         *        when coercion from {@link StructValue} is applied.
         * @return {@link ErrorValue} object.
         */
        private ErrorValue coerceErrorValue(String errName) {
            DataValue valueToConvert = value;
            if (convertDynamically(true)) {
                // Empty string like <NotFoundError></NotFoundError>
                // could represent error with no fields
                valueToConvert = new ErrorValue(errName);
            } else if (!(ErrorValue.class.isInstance(valueToConvert))
                       && StructValue.class.isInstance(valueToConvert)) {
                     // Above check for StructValue is implied
                // REST serializer does not know if we need struct
                // or error if we get StructValue convert it to ErrorValue
                ErrorValue ev = new ErrorValue(errName);
                StructValue sv = (StructValue) valueToConvert;
                for (String name : sv.getFieldNames()) {
                    ev.setField(name, sv.getField(name));
                }
                valueToConvert = ev;
            }
            return ConvertUtil.narrowType(valueToConvert, ErrorValue.class);
        }

        private String getAsString(DataValue value) {
            return ConvertUtil.narrowType(value, StringValue.class).getValue();
        }
    }

    /**
     * Visitor which converts a Java object to a vAPI data value in compliance
     * with the Java language bindings. This visitor should be accepted by a
     * type object which describes the expected output vAPI data value.
     */
    private class JavaToValueVisitor implements TypeVisitor {

        /**
         * Type converter needed to feed single-type converters.
         */
        private final TypeConverter typeConverter;

        /**
         * Java object which will be converted to a vAPI data value.
         */
        private final Object obj;

        /**
         * Output of the visitor.
         */
        private DataValue value;

        /**
         * Contextual data about the request to be used in serialization
         */
        private ConversionContext cc;

        /**
         * @param typeConverter type converter
         * @param obj   Java object which will be converted to a vAPI data
         *              value.
         * @param cc    Contextual data about the request to be used in
         *              serialization
         */
        public JavaToValueVisitor(TypeConverter typeConverter,
                                  Object obj,
                                  ConversionContext cc) {
            this.typeConverter = typeConverter;
            this.obj = obj;
            this.cc = cc;
        }

        /**
         * @return the output of the visitor: vAPI data value which corresponds
         *         to the type that accepted the visitor.
         */
        public DataValue getValue() {
            return value;
        }

        /**
         * Convert the Java language binding object to <code>VoidValue</code>.
         */
        @Override
        public void visit(VoidType type) {
            value = converterFactory.getVoidConverter().toValue(obj);
        }

        /**
         * Convert the Java language binding object to <code>BooleanValue</code>.
         */
        @Override
        public void visit(BooleanType type) {
            value = converterFactory.getBooleanConverter().toValue(obj);
        }

        /**
         * Convert the Java language binding object to <code>IntegerValue</code>.
         */
        @Override
        public void visit(IntegerType type) {
            value = converterFactory.getIntegerConverter().toValue(obj);
        }

        /**
         * Convert the Java language binding object to <code>DoubleValue</code>.
         */
        @Override
        public void visit(DoubleType type) {
            value = converterFactory.getDoubleConverter().toValue(obj);
        }

        /**
         * Convert the Java language binding object to <code>StringValue</code>.
         */
        @Override
        public void visit(StringType type) {
            value = converterFactory.getStringConverter().toValue(obj);
        }

        /**
         * Convert the Java language binding object to <code>BlobValue</code>.
         */
        @Override
        public void visit(BinaryType type) {
            value = converterFactory.getBinaryConverter().toValue(obj);
        }

        /**
         * Convert the Java language binding object to <code>SecretValue</code>.
         */
        @Override
        public void visit(SecretType type) {
            value = converterFactory.getSecretConverter().toValue(obj);
        }

        /**
         * Convert the Java language binding object to <code>StringValue</code>
         * in XSD dateTime format in UTC, i.e. YYYY-MM-DDTHH:MM:SS.sssZ.
         */
        @Override
        public void visit(DateTimeType type) {
            value = converterFactory.getDateTimeConverter().toValue(obj);
        }

        /**
         * Convert the Java language binding object to <code>StringValue</code>.
         */
        @Override
        public void visit(UriType type) {
            value = converterFactory.getUriConverter().toValue(obj);
        }

        /**
         * Convert the Java language binding object to
         * <code>OptionalValue</code>.
         */
        @Override
        public void visit(OptionalType type) {
            value = converterFactory.getOptionalConverter().toValue(
                    obj, type, typeConverter, cc);
        }

        /**
         * Convert the Java language binding object to <code>ListValue</code>.
         */
        @Override
        public void visit(ListType type) {
            value = converterFactory.getListConverter().toValue(
                    obj, type, typeConverter, cc);
        }

        @Override
        public void visit(SetType type) {
            value = converterFactory.getSetConverter().toValue(
                    obj, type, typeConverter, cc);
        }

        @Override
        public void visit(MapType type) {
            value = converterFactory.getMapConverter().toValue(
                    obj, type, typeConverter, cc);
        }

        /**
         * Convert the Java language binding object to <code>StructValue</code>.
         */
        @Override
        public void visit(StructType type) {
            StructValue structValue = (StructValue) converterFactory
                    .getStructConverter().toValue(obj, type, typeConverter, cc);

            // validate all constraints defined over the structure
            type.validate(structValue);

            value = structValue;
        }

        /**
         * Convert the Java language binding object to <code>DataValue</code>.
         */
        @Override
        public void visit(OpaqueType type) {
            value = converterFactory.getOpaqueConverter().toValue(obj);
        }

        /**
         * Resolve the type reference. Visit the actual type that the reference
         * points to.
         */
        @Override
        public void visit(TypeReference<? extends Type> type) {
            Type referredType = type.resolve();
            referredType.accept(this);
        }

        /**
         * Convert the Java language binding object to <code>StringValue</code>
         * which holds the name of an enumeration constant.
         */
        @Override
        public void visit(EnumType type) {
            value = converterFactory.getEnumConverter()
                    .toValue(obj, type, typeConverter, cc);
        }

        /**
         * Verify that the Java language binding object is a
         * {@link RuntimeException} and convert it to
         * an <code>ErrorValue</code>.
         */
        @Override
        public void visit(ErrorType type) {
            StructValue errorValue = (StructValue) converterFactory
                    .getErrorConverter().toValue(obj, type, typeConverter, cc);

            // validate all constraints defined over the error
            type.validate(errorValue);

            value = errorValue;
        }

        @Override
        public void visit(IdType idType) {
            value = converterFactory.getIdConverter().toValue(obj);
        }

        @Override
        public void visit(DynamicStructType type) {
            value = converterFactory.getDynamicStructureConverter().toValue(
                    obj, type, typeConverter, cc);

            // validate all constraints defined over the structure
            type.validate(ConvertUtil.narrowType(value, StructValue.class),
                          reusableThis());
        }

        @Override
        public void visit(AnyErrorType type) {
            value = converterFactory.getAnyErrorConverter().toValue(
                    obj, type, typeConverter, cc);
        }
    }

    private final ConverterFactory converterFactory;
    private final boolean reuseForValidation;

    /**
     * Default constructor. Uses default converter factory.
     * @see #TypeConverterImpl(ConverterFactory, boolean, boolean)
     */
    public TypeConverterImpl() {
        this(new DefaultConverterFactory());
    }

    /**
     * Constructor. Uses the specified converter factory.
     *
     * @param converterFactory factory {@code DataValue} converters
     * @see #TypeConverterImpl(ConverterFactory, boolean, boolean)
     */
    public TypeConverterImpl(ConverterFactory converterFactory) {
        this(converterFactory, true, false);
    }

    /**
     * Constructor. Uses default converter factory.
     * <p>
     * Deprecated, {@code TypeConverterImpl} is always in permissive mode now.
     *
     * @param permissive be more permissive when converting {@code DataValue}.
     *        Deprecated, {@code TypeConverterImpl} is always in permissive mode
     *        now
     * @see #TypeConverterImpl(ConverterFactory, boolean, boolean)
     */
    public TypeConverterImpl(boolean permissive) {
        this(new DefaultConverterFactory(), true, false);
    }

    /**
     * Constructor.
     *
     * @param converterFactory factory for converters for the various
     *        {@code DataValue} subtypes
     * @param permissive this type of conversion is used only when converting
     *        dynamic to static structure. the dynamic mode is more permissive
     *        by allowing the basic types to be represented as a
     *        {@link StringValue}. For {@link StructValue} and {@link ListValue}
     *        only an empty {@link StringValue} is accepted. For all other
     *        DataValues the {@link StringValue} should not be empty.
     *        Deprecated, {@code TypeConverterImpl} is always in permissive mode
     *        now
     * @param reuseForValidation whether this instance must be reused for
     *        validation and conversion purposes
     */
    public TypeConverterImpl(ConverterFactory converterFactory,
                             boolean permissive,
                             boolean reuseForValidation) {
        Validate.notNull(converterFactory);
        this.converterFactory = converterFactory;
        this.reuseForValidation = reuseForValidation;
    }

    @Override
    public <T> T convertToJava(DataValue value,
                               Type type) {

        if (!(type instanceof OptionalType) && value == null) {
            throw new ConverterException(
                    "vapi.bindings.typeconverter.fromvalue.value.missing");
        }
        if (type == null) {
            throw new ConverterException(
                    "vapi.bindings.typeconverter.fromvalue.type.missing");
        }
        ValueToJavaVisitor visitor = new ValueToJavaVisitor(this, value);
        type.accept(visitor);
        return visitor.<T>getJavaObject();
    }

    @Override
    public DataValue convertToVapi(Object obj,
                                   Type type) {

        return this.convertToVapi(obj, type, new ConversionContext());
    }

    @Override
    public DataValue convertToVapi(Object obj, Type type, ConversionContext cc) {
        if (type == null) {
            throw new ConverterException(
                    "vapi.bindings.typeconverter.tovalue.type.missing");
        }
        JavaToValueVisitor visitor = new JavaToValueVisitor(this, obj, cc);
        type.accept(visitor);
        return visitor.getValue();
    }

    @Override
    public TypeConverter reusableThis() {
        if (reuseForValidation) {
            return this;
        }

        return null;
    }
}
