/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc. All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings;

import com.vmware.vapi.bindings.type.ListType;
import com.vmware.vapi.bindings.type.OptionalType;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.bindings.type.TypeReference;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.data.ConstraintValidationException;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.util.Validate;

/**
 * Utility class to help enforcing the validation rules over the data values
 */
public final class ValidatorUtil {

    /**
     * Enforces recursively all static (defined in the IDL) or dynamic
     * constrains defined over the type.
     *
     * @param type can be <code>null</code>
     * @param value can be <code>null</code>
     * @param methodId the method invocation in which context the dataValue was
     *                 passed (as a parameter). must not be <code>null</code>
     * @throws ConstraintValidationException if the constraint is not satisfied
     */
    public static void validate(Type type, DataValue value, MethodIdentifier methodId) {
        Validate.notNull(methodId);
        if (type instanceof TypeReference<?>) {
            validate(((TypeReference<?>) type).resolve(), value, methodId);
        } else if (type instanceof OptionalType) {
            if (value instanceof OptionalValue && ((OptionalValue)value).isSet()) {
                validate(((OptionalType)type).getElementType(),
                         ((OptionalValue)value).getValue(),
                         methodId);
            }
        } else if (type instanceof ListType && value instanceof ListValue) {
            ListValue listValue = (ListValue)value;
            for (DataValue dataValue : listValue) {
                validate(((ListType)type).getElementType(), dataValue, methodId);
            }
        } else if (type instanceof StructType && value instanceof StructValue) {
            StructType structType = (StructType)type;
            StructValue structValue = (StructValue)value;
            structType.validate(structValue, methodId);
            // validate recursively
            for (String fieldName : structType.getFieldNames()) {
                DataValue fieldValue = (structValue.hasField(fieldName))?
                        structValue.getField(fieldName) : null;
                validate(structType.getField(fieldName), fieldValue, methodId);
            }
        }
    }
}
