/* **********************************************************
 * Copyright (c) 2013-2014, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings.convert.impl;

import static com.vmware.vapi.MessageFactory.getMessage;

import java.lang.reflect.Method;

import com.vmware.vapi.bindings.ApiEnumeration;
import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.bindings.type.EnumType;
import com.vmware.vapi.bindings.type.StringType;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.bindings.TypeConverter.ConversionContext;
import com.vmware.vapi.internal.bindings.convert.UniTypeConverter;

/**
 * Convert enumeration type to a code-generated binding class which implements
 * {@link ApiEnumeration}. The binding class is expected to have a static
 * method valueOf() which accepts a string and returns an instance of the
 * binding class.
 */
public class ApiEnumerationEnumConverter implements
        UniTypeConverter<StringValue, EnumType> {

    @Override
    public Object fromValue(StringValue value,
                            EnumType declaredType,
                            TypeConverter typeConverter) {
        Class<? extends ApiEnumeration<?>> enumClass =
                verifyBindingClass(declaredType.getBindingClass());
        try {
            Method valueOf =
                    enumClass.getDeclaredMethod("valueOf", String.class);
            Object obj = valueOf.invoke(null, value.getValue());
            return enumClass.cast(obj);
        } catch (Exception ex) {
            throw new ConverterException(getMessage(
                    "vapi.bindings.typeconverter.enum.valueof.error",
                    enumClass.getCanonicalName(), value.getValue()), ex);
        }
    }

    @Override
    public DataValue toValue(Object binding,
                             EnumType declaredType,
                             TypeConverter typeConverter,
                             ConversionContext cc) {
        Class<? extends ApiEnumeration<?>> enumClass =
                verifyBindingClass(declaredType.getBindingClass());
        // verify that the binding object is of the expected binding class
        ApiEnumeration<?> e = ConvertUtil.narrowType(binding, enumClass);
        return typeConverter.convertToVapi(e.name(), new StringType(), cc);
    }

    /**
     * Verifies that the specified binding class represents a
     * {@link ApiEnumeration}.
     *
     * @param c Java class
     */
    @SuppressWarnings({ "unchecked" })
    private Class<? extends ApiEnumeration<?>> verifyBindingClass(Class<?> c) {
        if (ApiEnumeration.class.isAssignableFrom(c)) {
            return (Class<? extends ApiEnumeration<?>>) c;
        } else {
            throw new ConverterException(getMessage(
                    "vapi.bindings.typeconverter.enum.invalid.binding.class",
                    c.getCanonicalName()));
        }
    }
}
