/* **********************************************************
 * Copyright (c) 2012-2015, 2018-2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings.convert.impl;

import java.lang.reflect.Method;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.bindings.StaticStructure;
import com.vmware.vapi.bindings.Structure;
import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.bindings.type.OptionalType;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.bindings.BindingsUtil;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.bindings.TypeConverter.ConversionContext;
import com.vmware.vapi.internal.bindings.convert.UniTypeConverter;
import com.vmware.vapi.internal.util.Validate;

/**
 * Convert {@code StructValue} or {@code ErrorValue} instance to and instance of
 * code-generated bindings Java class.
 */
public class JavaClassStructConverter<V extends StructValue, T extends StructType>
        implements UniTypeConverter<V, T> {

    private static final Logger logger = LoggerFactory
            .getLogger(JavaClassStructConverter.class);

    private final Class<V> valueClass;
    private final Class<? extends Structure> typeClass;
    private final boolean skipUnsetOptionalFields;

    /**
     * Creates a converter for structures or errors.
     *
     * @param valueClass the StructValue class, which must be StructValue.class
     *        to convert structures and ErrorValue.class to convert errors; must
     *        not be <code>null</code>
     * @param typeClass the StructType class, which must be Structure.class to
     *        convert structures and ApiError.class to convert errors; must not
     *        be <code>null</code>
     * @param skipUnsetOptionalFields if set to {@code true} the unset optional
     *        fields will be completely skipped in the results of
     *        {@link #toValue()}
     */
    public JavaClassStructConverter(Class<V> valueClass,
                                    Class<? extends Structure> typeClass,
                                    boolean skipUnsetOptionalFields) {
        Validate.notNull(valueClass);
        Validate.notNull(typeClass);
        this.valueClass = valueClass;
        this.typeClass = typeClass;
        this.skipUnsetOptionalFields = skipUnsetOptionalFields;
    }

    /**
     * Delegates to {@link #JavaClassStructConverter(Class, Class, boolean)}
     * with {@code false} value for {@code skipUnsetOptionalFields}.
     */
    public JavaClassStructConverter(Class<V> valueClass,
                                    Class<? extends Structure> typeClass) {
        this(valueClass, typeClass, false);
    }

    @Override
    public Object fromValue(V value,
                            T declaredType,
                            TypeConverter typeConverter) {
        final StaticStructure struct = createStructBinding(declaredType, value);
        for (String fieldName : declaredType.getFieldWithSetterNames()) {
            Type fieldType = declaredType.getField(fieldName);
            DataValue fieldValue = null;
            if (value.hasField(fieldName)) {
                fieldValue = value.getField(fieldName);
            }

            try {
                Object fieldObj = typeConverter.convertToJava(fieldValue,
                                                              fieldType);
                BindingsUtil.setStructureFieldValue(struct,
                                                    declaredType,
                                                    fieldName,
                                                    fieldObj);
            } catch (RuntimeException ex) {
                throw new ConverterException(MessageFactory
                        .getMessage("vapi.bindings.typeconverter.fromvalue.struct.field.error",
                                    fieldName,
                                    declaredType.getName()), ex);
            }
        }
        return struct;
    }

    @Override
    public DataValue toValue(Object binding,
                             T declaredType,
                             TypeConverter typeConverter,
                             ConversionContext cc) {
        StaticStructure struct = checkBindingType(binding, declaredType);
        StructValue result = createStructInstance(declaredType.getName());
        // set static fields
        for (String fieldName : declaredType.getFieldNames()) {
            Object fieldBinding = extractFieldValue(declaredType,
                                                    struct,
                                                    fieldName,
                                                    cc);
            Type fieldType = declaredType.getField(fieldName);
            if (fieldBinding == null) {
                checkMissingRequiredField(fieldName, fieldType, declaredType);

                if (skipUnsetOptionalFields) {
                    // skip this optional field with "null" value altogether
                    continue;
                }
            }
            readFieldDataValue(fieldName,
                               fieldBinding,
                               fieldType,
                               result,
                               typeConverter,
                               declaredType,
                               cc);
        }

        return processDynamicFields(struct, result);
    }

    protected Object extractFieldValue(T declaredType,
                                       StaticStructure struct,
                                       String fieldName,
                                       ConversionContext cc) {
        Object fieldBinding = BindingsUtil
                .getStructureFieldValue(struct, declaredType, fieldName);
        return fieldBinding;
    }

    /**
     * Returns if unset optional fields will be output in the resulting data
     * value
     */
    protected final boolean isSkipUnsetOptionalFields() {
        return skipUnsetOptionalFields;
    }

    protected static void checkMissingRequiredField(String fieldName,
                                                    Type fieldType,
                                                    StructType declaredType) {
        if (!(fieldType instanceof OptionalType)) {
            // missing required field
            throw new ConverterException(MessageFactory
                    .getMessage("vapi.bindings.typeconverter.tovalue.struct.field.missing",
                                fieldName,
                                declaredType.getName()));
        }
    }

    protected StaticStructure checkBindingType(Object binding,
                                               StructType declaredType) {
        StaticStructure struct = ConvertUtil.narrowType(binding,
                                                        StaticStructure.class);
        checkStructBinding(declaredType, struct);
        return struct;
    }

    protected static void readFieldDataValue(String fieldName,
                                             Object fieldBinding,
                                             Type fieldType,
                                             StructValue result,
                                             TypeConverter typeConverter,
                                             StructType declaredType,
                                             ConversionContext cc) {
        try {
            DataValue dataValue = typeConverter
                    .convertToVapi(fieldBinding, fieldType, cc);
            result.setField(fieldName, dataValue);
        } catch (RuntimeException ex) {
            throw new ConverterException(MessageFactory
                    .getMessage("vapi.bindings.typeconverter.tovalue.struct.field.error",
                                fieldName,
                                declaredType.getName()),
                                ex);
        }
    }

    protected static DataValue processDynamicFields(final StaticStructure input,
                                           StructValue result) {
        if ("static".equals(System
                .getProperty("com.vmware.vapi.bindings.SerializationPolicy"))) {
            // skip the dynamic fields; return now (only static fields) for
            // "static serialization policy"
            return result;
        }

        // set dynamic fields
        for (String fieldName : input._getDynamicFieldNames()) {
            DataValue dataValue = input._getDynamicField(fieldName);
            result.setField(fieldName, dataValue);
        }

        return result;
    }

    /**
     * Extracts the dynamic fields for the given structType to separate
     * structure.
     *
     * @param structType the structure type
     * @param value the structure value that contains the dynamic fields
     * @return structure value containing only the dynamic fields (i.e. fields
     *         missing in the specified structure type) or null if there are no
     *         dynamic fields.
     */
    private StructValue extractDynamicFields(StructType structType,
                                             StructValue value) {
        StructValue dynamicFieldsStructValue = null;
        Set<String> staticFields = structType.getFieldNames();

        for (String fieldName : value.getFieldNames()) {
            if (!staticFields.contains(fieldName)) {
                if (dynamicFieldsStructValue == null) {
                    dynamicFieldsStructValue =
                            createStructInstance(value.getName());
                }
                dynamicFieldsStructValue.setField(fieldName,
                                                  value.getField(fieldName));
            }
        }

        return dynamicFieldsStructValue;
    }

    /**
     * Creates a non-initialized binding for the specified structure type.
     *
     * @param structType structure type
     * @param value the initial value of the structure
     * @return structure binding
     */
    protected StaticStructure createStructBinding(StructType structType,
                                                StructValue value) {
        Class<? extends Structure> structClass = getStructClass(structType);

        try {
            Method newInstance = null;
            StructValue dynamicFields = extractDynamicFields(structType, value);
            try {
                newInstance = structClass.getMethod("_newInstance2",
                                                    StructValue.class);
                // we need to propagate the structure type name to the binding
                if (dynamicFields == null
                        && !structType.getName().equals(value.getName())) {
                    dynamicFields = createStructInstance(value.getName());
                }
            } catch (NoSuchMethodException e) {
                logger.trace("Old binding for type {} detected. Resorting to "
                             + "memory inefficient instantiation",
                             structType.getName());
                newInstance = structClass.getMethod("_newInstance",
                                                    StructValue.class);
                // null is not allowed by older bindings
                if (dynamicFields == null) {
                    dynamicFields = createStructInstance(value.getName());
                }
            }
            return (StaticStructure) newInstance.invoke(null, dynamicFields);
        } catch (Exception ex) {
            Message message = MessageFactory
                    .getMessage("vapi.bindings.structbinding.struct.ctor.error",
                                structType.getName(),
                                structClass.getCanonicalName());
            logger.error(message.toString());
            throw new ConverterException(message, ex);
        }
    }

    /**
     * Verifies that the structure binding matches the structure type.
     *
     * @param structType structure type
     * @param struct structure binding
     * @throws ConverterException if the binding does not match the type
     */
    private void checkStructBinding(StructType structType, Structure struct) {
        Class<? extends Structure> expectedClass = getStructClass(structType);

        if (!expectedClass.isInstance(struct)) {
            throw new ConverterException("vapi.bindings.structbinding.unexpected.binding.class",
                                         expectedClass.getCanonicalName(),
                                         structType.getName(),
                                         struct.getClass().getCanonicalName());
        }
    }

    /**
     * Obtains and validates the structure binding class from the type
     * information.
     *
     * @param structType structure type
     * @return structure binding class
     */
    private Class<? extends Structure> getStructClass(StructType structType) {
        if (!typeClass.isAssignableFrom(structType.getBindingClass())) {
            Message message = MessageFactory
                    .getMessage("vapi.bindings.structbinding.inconsistent.type.info",
                                structType.getBindingClass()
                                        .getCanonicalName());
            logger.error(message.toString());
            throw new ConverterException(message);
        }
        @SuppressWarnings("unchecked")
        Class<? extends Structure> structClass = (Class<? extends Structure>) structType
                .getBindingClass();
        return structClass;
    }

    protected V createStructInstance(String name) {
        try {
            return valueClass.getConstructor(String.class).newInstance(name);
        } catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }
}
