/* **********************************************************
 * Copyright 2012-2013 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * JavaNetUriUriConverter.java --
 *
 *      Convert URI type to java.net.URI
 */
package com.vmware.vapi.internal.bindings.convert.impl;

import static com.vmware.vapi.MessageFactory.getMessage;

import java.net.URI;
import java.net.URISyntaxException;

import com.vmware.vapi.data.ConstraintValidationException;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.internal.bindings.convert.PrimitiveConverter;

/**
 * <p>
 * Convert URI type to java.net.URI. The URI type is represented by
 * <code>StringValue</code> in the API runtime, so this converter converts
 * between URI string and <code>java.net.URI</code>.
 * </p>
 * <p>
 * The URI string representation must conform to the IRI specification as
 * defined by RFC 3987. Unlike the URI specification RFC 3986, the IRI
 * specification does not require percent-encoding of non-ASCII Unicode symbols.
 * Even though it is not mandatory, percent-encoding is still allowed. When
 * percent-encoding, UTF-8 must be used to represent the Unicode character as
 * octets.
 * </p>
 */
public final class JavaNetUriUriConverter implements PrimitiveConverter<StringValue> {

    @Override
    public URI fromValue(StringValue value) {
        String iriString = value.getValue();
        try {
            return new URI(iriString);
        } catch (URISyntaxException ex) {
            throw new ConstraintValidationException(
                    getMessage("vapi.bindings.typeconverter.uri.invalid.format",
                               iriString),
                    ex);
        }
    }

    @Override
    public StringValue toValue(Object primitiveBinding) {
        URI uri = ConvertUtil.narrowType(primitiveBinding, URI.class);
        return new StringValue(uri.toString());
    }

}
