/* **********************************************************
 * Copyright (c) 2012-2014, 2017-2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings.convert.impl;

import static com.vmware.vapi.internal.bindings.convert.Constants.MAP_TYPE_ENTRY_KEY_FIELD;
import static com.vmware.vapi.internal.bindings.convert.Constants.MAP_TYPE_ENTRY_VALUE_FIELD;
import static com.vmware.vapi.internal.bindings.convert.impl.JavaUtilMapStructValueMapConverter.convertStructValueToMap;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.bindings.type.MapType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.bindings.TypeConverter.ConversionContext;
import com.vmware.vapi.internal.bindings.convert.UniTypeConverter;
import com.vmware.vapi.std.BuiltInDataFactory;

/**
 * Converts a {@code DataValue} represented map to a {@code java.util.Map}.
 *
 * <p>
 * The actual implementation used for the native map type is {@link HashMap}.
 *
 * <p>
 * The map is assumed to be represented as a {@code ListValue} of
 * {@code StructValue} instances for the map entries. Each of the latter is
 * expected to have {@code "key"} and {@code "value"} fields, holding the values
 * of the map entry.
 */
public final class JavaUtilMapMapConverter implements
        UniTypeConverter<DataValue, MapType> {

    private static final Set<String> MAP_ENTRY_VALUE_FIELDS =
            new HashSet<>(Arrays.asList(MAP_TYPE_ENTRY_KEY_FIELD,
                                        MAP_TYPE_ENTRY_VALUE_FIELD));

    @Override
    public Object fromValue(DataValue value,
                            MapType declaredType,
                            TypeConverter typeConverter) {
        if(value instanceof ListValue) {
            return convertListToMap(value, declaredType, typeConverter);
        } else {
            return convertStructValueToMap(value, declaredType, typeConverter);
        }
    }

    @Override
    public DataValue toValue(Object binding,
                             MapType declaredType,
                             TypeConverter typeConverter,
                             ConversionContext cc) {
        @SuppressWarnings("unchecked")
        Map<Object, Object> mapBinding = ConvertUtil.narrowType(binding, Map.class);
        Type keyType = declaredType.getKeyType();
        Type valueType = declaredType.getValueType();
        ListValue mapValue = new ListValue();
        for (Map.Entry<Object, Object> elem : mapBinding.entrySet()) {
            StructValue entry = new StructValue(BuiltInDataFactory.MAP_ENTRY_STRUCT_NAME);
            DataValue key = typeConverter
                    .convertToVapi(elem.getKey(), keyType, cc);
            DataValue val = typeConverter
                    .convertToVapi(elem.getValue(), valueType, cc);
            entry.setField(MAP_TYPE_ENTRY_KEY_FIELD, key);
            entry.setField(MAP_TYPE_ENTRY_VALUE_FIELD, val);
            mapValue.add(entry);
        }
        return mapValue;
    }

    private static Object convertListToMap(DataValue value,
                                           MapType declaredType,
                                           TypeConverter typeConverter) {
        ListValue listValue = ConvertUtil.narrowType(value, ListValue.class);

        Type keyType = declaredType.getKeyType();
        Type valueType = declaredType.getValueType();

        Map<Object, Object> mapBinding = new HashMap<>();
        for (DataValue elementVal : listValue) {
            if (!(elementVal instanceof StructValue) ) {
                throw new ConverterException(
                        "vapi.bindings.typeconverter.map." +
                        "unexpected.key.value.entry");
            }
            StructValue structVal = (StructValue) elementVal;
            if (!MAP_ENTRY_VALUE_FIELDS.equals(structVal.getFieldNames())) {
                throw new ConverterException(
                "vapi.bindings.typeconverter.map.unexpected.key.value.entry");
            }
            Object key = typeConverter.convertToJava(
                    structVal.getField(MAP_TYPE_ENTRY_KEY_FIELD), keyType);
            Object val = typeConverter.convertToJava(
                    structVal.getField(MAP_TYPE_ENTRY_VALUE_FIELD),
                    valueType);

            if (mapBinding.containsKey(key)) {
                throw new ConverterException(
                        "vapi.bindings.typeconverter.map.duplicate.element");
            }
            mapBinding.put(key, val);
        }
        return mapBinding;
    }
}
