/* **********************************************************
 * Copyright (c) 2013, 2017-2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings.convert.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.vmware.vapi.bindings.type.ErrorType;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.internal.bindings.convert.NameToTypeResolver;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.util.StructTypeUtil;

/**
 * Resolver which is backed by a map.
 */
public final class MapBasedNameToTypeResolver implements NameToTypeResolver {

    /**
     * Type map which contains all standard errors.
     */
    public static final Map<String, StructType> STANDARD_ERRORS;
    static {
        // load all standard types
        Map<String, StructType> typeMap = new HashMap<>();
        StructTypeUtil.populateCanonicalNameToStructTypeMap(typeMap);
        // filter only the error types
        Map<String, StructType> errorMap = new HashMap<>();
        for (Map.Entry<String, StructType> e : typeMap.entrySet()) {
            if (e.getKey().startsWith("com.vmware.vapi.std.errors")) {
                errorMap.put(e.getKey(), e.getValue());
            }
        }
        STANDARD_ERRORS = Collections.unmodifiableMap(errorMap);
    }

    private final Map<String, StructType> typeMap;

    /**
     * Creates a resolver which is backed by the specified map instance. Note
     * that the map is used directly and is not copied.
     *
     * @param typeMap type map; must not be <code>null</code>
     */
    public MapBasedNameToTypeResolver(Map<String, StructType> typeMap) {
        Validate.notNull(typeMap);
        Validate.noNullElements(typeMap.keySet());
        Validate.noNullElements(typeMap.values());
        this.typeMap = Collections.unmodifiableMap(typeMap);
    }

    @Override
    public Type resolve(String name) {
        Validate.notNull(name);
        return typeMap.get(name);
    }

    /**
     * Augments specified set of errors with standard errors.
     *
     * @param typeSet set of custom error types; may be {@code null} or empty
     * @return union of the custom errors and the standard errors
     */
    public static Map<String, StructType> augmentWithStandardErrors(Set<ErrorType> typeSet) {
        if(typeSet == null || typeSet.isEmpty()) {
            return MapBasedNameToTypeResolver.STANDARD_ERRORS;
        }

        Map<String, StructType> errorTypeMap = new HashMap<>();
        errorTypeMap.putAll(MapBasedNameToTypeResolver.STANDARD_ERRORS);
        errorTypeMap.putAll(toMap(typeSet));

        return errorTypeMap;
    }

    private static Map<String, StructType> toMap(Set<ErrorType> typeSet) {
        Map<String, StructType> typeMap = new HashMap<>();
        for (ErrorType t : typeSet) {
            typeMap.put(t.getName(), t);
        }
        return typeMap;
    }
}
