/* **********************************************************
 * Copyright (c) 2012-2014, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings.server.impl;

import java.util.Collections;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.bindings.server.AsyncContext;
import com.vmware.vapi.bindings.server.InvocationContext;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.internal.bindings.ApiMethodSkeleton;
import com.vmware.vapi.internal.bindings.ProgressConverter;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.bindings.TypeConverter.ConversionContext;
import com.vmware.vapi.std.Progress;
import com.vmware.vapi.std.StandardDataFactory;

/**
 * This utility adapts an {@link AsyncHandle} from the runtime to the
 * {@link AsyncContext} interface of the bindings.
 */
public final class AsyncContextImpl<T> extends AsyncContext<T> {

    private static final Logger logger =
            LoggerFactory.getLogger(AsyncContextImpl.class);

    private final InvocationContext invocationContext;

    private final AsyncHandle<MethodResult> asyncHandle;

    private final TypeConverter typeConverter;

    private final Type outputType;

    private final ApiMethodSkeleton method;

    public AsyncContextImpl(TypeConverter typeConverter,
                            Type outputType,
                            InvocationContext invocationContext,
                            AsyncHandle<MethodResult> asyncHandle,
                            ApiMethodSkeleton method) {
        this.typeConverter = typeConverter;
        this.outputType = outputType;
        this.invocationContext = invocationContext;
        this.asyncHandle = asyncHandle;
        this.method = method;
    }

    @Override
    public InvocationContext getInvocationContext() {
        return invocationContext;
    }

    @Override
    public void updateProgress(Progress progress) {
        DataValue progressValue = ProgressConverter.toValue(progress);
        asyncHandle.updateProgress(progressValue);
    }

    @Override
    public void setResult(T result) {
        DataValue output;
        try {
            ExecutionContext ec = invocationContext.getExecutionContext();
            ConversionContext cc = new ConversionContext(ec);
            output = typeConverter.convertToVapi(result, outputType, cc);
        } catch (RuntimeException e) {
            setError(e);
            return;
        }
        asyncHandle.setResult(MethodResult.newResult(output));
    }

    @Override
    public void setError(RuntimeException error) {
        ErrorValue value;
        try {
            if (error != null) {
                ExecutionContext ec = invocationContext
                        .getExecutionContext();
                value = method.toErrorValue(typeConverter, error, ec);
            } else {
                value = createDefaultErrorValue(null);
            }
        } catch (RuntimeException e) {
            value = createDefaultErrorValue(e);
        }
        asyncHandle.setResult(MethodResult.newErrorResult(value));
    }

    private ErrorValue createDefaultErrorValue(RuntimeException e) {
        if (e == null) {
            logger.debug("Unknown exception occured",
                         new RuntimeException("For stack trace purpose"));
        } else {
            logger.debug("Unexpected exception occured", e);
        }
        ErrorValue value = StandardDataFactory.createErrorValueForMessages(
                StandardDataFactory.INTERNAL_SERVER_ERROR,
                Collections.singletonList(MessageFactory.getMessage(
                        "vapi.bindings.asynccontext.error.convert",
                        String.valueOf(e.getMessage()))));
        return value;
    }
}
