/* **********************************************************
 * Copyright 2012-2015, 2019 VMware, Inc. All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.data;

import static com.vmware.vapi.MessageFactory.getMessage;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.data.StructDefinition;
import com.vmware.vapi.data.StructRefDefinition;
import com.vmware.vapi.internal.util.Validate;

/**
 * Context which keeps track of structure definitions and structure references
 * in a graph of definition objects. The context can resolve the unresolved
 * references after it learns about all definitions and references in the
 * definition graph.
 */
public class ReferenceResolver {

    private final Map<String, StructDefinition> definitions =
        new HashMap<String, StructDefinition>();

    private final List<StructRefDefinition> references =
        new ArrayList<StructRefDefinition>();

    /**
     * Adds a new structure definition to the context.
     *
     * @param structDef definition to be added; must not be {@code null}.
     */
    public void addDefinition(StructDefinition structDef) {
        Validate.notNull(structDef);
        definitions.put(structDef.getName(), structDef);
    }

    /**
     * Adds a new structure definition reference to the context.
     *
     * @param structRefDef structure definition reference; must not be {@code null}.
     */
    public void addReference(StructRefDefinition structRefDef) {
        Validate.notNull(structRefDef);
        references.add(structRefDef);
    }

    /**
     * Determines whether the context contains a definition for the specified
     * structure.
     *
     * @param structName structure name; must not be {@code null}.
     * @return whether the context contains definition of the structure.
     */
    public boolean isDefined(String structName) {
        Validate.notNull(structName);
        return definitions.containsKey(structName);
    }

    /**
     * Traverses all references and resolves the unresolved ones.
     *
     * @throws CoreException if a reference cannot be resolved.
     */
    public void resolveReferences() {
        for (StructRefDefinition ref : references) {
            if (ref.getTarget() == null) {
                StructDefinition def = definitions.get(ref.getName());
                if (def == null) {
                    throw new CoreException(getMessage(
                        "vapi.data.structref.structure.not.defined",
                        ref.getName()));
                }
                ref.setTarget(def);
            }
        }
    }

    /**
     * To obtain the definition of a structure if it was already defined.
     *
     * @param structName structure name; must not be {@code null}.
     * @return definition of the structure with the name {@code structName};
     *         {@code null} otherwise.
     */
    public StructDefinition getDefinition(String structName) {
        Validate.notNull(structName);
        return definitions.get(structName);
    }
}
