/* **********************************************************
 * Copyright (c) 2012-2015, 2016, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.data;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.data.ConstraintValidationException;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.util.Validate;

/**
 * Validator for discriminated unions state.
 *
 * <p>Each instance is capable to validate the state/consistency for a single discriminated
 * union in a <code>StructValue</code>. For a detailed validation description refer to the
 * {@link #validate(DataValue)} method javadoc.
 *
 * <p>This implementation tolerates the absence of fields in the {@code StructValue} being
 * validated and treats such fields as unset {@code OptionalValue} instances.
 */
public class UnionValidator implements ConstraintValidator {

    private final String discriminantFieldName;
    private final Map<String, List<FieldData>> caseFields;
    private final Set<FieldData> allUnionFields;

    /**
     * Constructor.
     *
     * @param discriminantFieldName name of the field that is discriminant/tag
     *                              for the union; may not be null
     * @param caseFields map of discriminant value as key (in canonical form) ->
     *                  list of {@link FieldData} for that discriminant value;
     *                  must not be <code>null</code>. Lists of {@link FieldData}
     *                  must not be <code>null</code>
     * @throws IllegalArgumentException if <code>discriminantFieldName</code>
     *                                  or <code>caseFields</code> is null or
     *                                  any List<FieldData> is <code>null</code>
     */
    public UnionValidator(String discriminantFieldName,
                          Map<String, List<FieldData>> caseFields) {
        Validate.notNull(discriminantFieldName);
        Validate.notNull(caseFields);
        for (List<FieldData> fieldDataList : caseFields.values()) {
            Validate.notNull(fieldDataList);
        }

        this.discriminantFieldName = discriminantFieldName;
        this.caseFields = caseFields;

        this.allUnionFields = new HashSet<>();
        for (Map.Entry<String, List<FieldData>> entry : caseFields.entrySet()) {
            this.allUnionFields.addAll(entry.getValue());
        }
    }

    /**
     * Validates the state of the discriminated unions represented by
     * this instance in the specified structure <code>struct</code>.
     *
     * <p>More precisely, given value of the discriminant/tag field from
     * <code>struct</code>, validates that:
     *  <li>all fields of the union required for that value of the
     *      discriminant are set except if a field is declared as Optional
     *  <li>fields of the union that are not allowed for that value of the
     *      discriminant are not set
     *  <li>any other fields of the structure (not participating in the union)
     *      do not affect the validation result
     *
     * @param dataValue  struct value to validate; may not be {@code null}
     * @throws ConstraintValidationException if the discriminated unions is in invalid state
     * @throws IllegalArgumentException if {@code dataValue} is not a {@code StructValue} instance
     */
    @Override
    public void validate(DataValue dataValue) {
        Validate.isTrue(dataValue instanceof StructValue);
        StructValue structValue = (StructValue) dataValue;
        DataValue discriminantValue = null;
        if (structValue.hasField(discriminantFieldName)) {
            discriminantValue = structValue.getField(discriminantFieldName);
        }

        if (isMissingOrUnsetOptionalValue(discriminantValue)) {
            validateOtherFieldsAreNotSet(structValue, discriminantValue,
                  Collections.<FieldData>emptyList());
            return;
        }

        if (discriminantValue instanceof OptionalValue) {
            discriminantValue = ((OptionalValue) discriminantValue).getValue();
        }

        Validate.isTrue(discriminantValue instanceof StringValue);
        List<FieldData> expectedFields = caseFields.get(
                ((StringValue) discriminantValue).getValue());

        if (expectedFields == null) {
            expectedFields = Collections.<FieldData>emptyList();
        }

        validateExpectedFieldsAreSet(structValue, expectedFields);
        validateOtherFieldsAreNotSet(structValue, discriminantValue,
            expectedFields);
    }

    /**
     * Validates that the required union fields are either present OR defined as optional.<p/>
     * This method does not validate optional fields have OptionalValue container(See bug# 1499063)
     */
    void validateExpectedFieldsAreSet(StructValue struct,
                                      List<FieldData> expectedFields) {
        for (FieldData expField : expectedFields) {
            if(expField.isOptional()) {
                // Explicitly optional UnionCase fields need no checking
                continue;
            }

            DataValue fieldValue = null;
            if (struct.hasField(expField.getName())) {
                fieldValue = struct.getField(expField.getName());
            }

            if (isMissingOrUnsetOptionalValue(fieldValue)) {
                // validation failed - non optional field required by the discriminant
                // is missing or is not set
                throw new ConstraintValidationException(MessageFactory.getMessage(
                        "vapi.data.structure.union.missing",
                        struct.getName(),
                        expField.getName()));
            }
        }
    }

    void validateOtherFieldsAreNotSet(StructValue struct,
                                      DataValue discriminantValue,
                                      List<FieldData> expectedFields) {
        Set<FieldData> otherFields = new HashSet<>(allUnionFields);
        otherFields.removeAll(expectedFields);

        for (FieldData otherField : otherFields) {
            DataValue fieldValue = null;
            if (struct.hasField(otherField.getName())) {
                fieldValue = struct.getField(otherField.getName());
            }
            if (!isMissingOrUnsetOptionalValue(fieldValue)) {
                // validation failed - field not allowed by the discriminant is set
                Message errMessage = MessageFactory.getMessage(
                        "vapi.data.structure.union.extra",
                        struct.getName(),
                        discriminantFieldName,
                        discriminantValue == null ?
                                "<unset>" : discriminantValue.toString(),
                        otherField.getName());
                throw new ConstraintValidationException(errMessage);
            }
        }
    }

    private boolean isMissingOrUnsetOptionalValue(DataValue dataValue) {
        if (dataValue == null) {
            return true;  // missing
        }

        if (dataValue instanceof OptionalValue) {
            return !((OptionalValue) dataValue).isSet();
        }

        // it is not if its type is different
        return false;
    }

    /**
     * This class represents all field data that is needed for union validation
     */
    public static final class FieldData {

        private final String name;
        private final boolean isOptional;

        public FieldData(String name, boolean isOptional) {
            Validate.notEmpty(name);
            this.name = name;
            this.isOptional = isOptional;
        }

        /**
         * @return name of the field. cannot be <code>null</code>
         */
        public String getName() {
            return name;
        }

        /**
         * @return <code>true</code> if the field is not required for the union
         *         validation to succeed and <code>false</code> otherwise.
         */
        public boolean isOptional() {
            return isOptional;
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + (isOptional ? 1231 : 1237);
            result = prime * result + name.hashCode();
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            FieldData other = (FieldData) obj;
            if (isOptional != other.isOptional)
                return false;
            if (!name.equals(other.name))
                return false;
            return true;
        }

        @Override
        public String toString() {
            return getClass().getSimpleName() +
                    "<" + this.name + ", " +
                    this.isOptional + ">";
        }
    }
}
