/* **********************************************************
 * Copyright (c) 2012, 2016, 2019 VMware, Inc. All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.dsig.json;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.TreeMap;

import com.fasterxml.jackson.core.JsonGenerationException;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.Version;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.node.DoubleNode;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;
import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.dsig.json.SignatureException;
import com.vmware.vapi.internal.util.Validate;

/**
 * This class is responsible for transforming JSON messages into their canonical
 * representation.<br>
 *<br>
 * TODO remove the following once the rfc is published and add a link to the rfc
 *<br><br>
 * The canonical form is defined by the following rules:
 * <ul>
 * <li>
 * The document is encoded in UTF-8 [UTF-8]</li>
 * <li>
 * Non-significant(1) whitespace characters MUST NOT be used</li>
 * <li>
 * Non-significant(1) line endings MUST NOT be used</li>
 * <li>
 * Entries (set of name/value pairs) in JSON objects MUST be sorted
 * lexicographically(2) by their names</li>
 * <li>
 * Arrays MUST preserve their initial ordering</li>
 * </ul>
 *
 * (1)Non-significant means not part of "name" or "value". Based on the JSON
 * data-interchange format JSON objects consists or multiple "name"/"value"
 * pairs and JSON arrays consists of multiple "value" fields.<br>
 *
 * (2)Lexicographic comparison, which orders strings from least to greatest
 * alphabetically based on the UCS (Unicode Character Set) codepoint values.<br>
 * <br>
 * The double data type is represented as specified in the XML schema standard
 * [XML]
 * <ul>
 * <li>
 * The canonical representation of the double data type consists of mantissa
 * followed by "E", followed by exponent.</li>
 * <li>
 * Mantissa</li>
 * <ul>
 * <li>
 * MUST be represented as a decimal. The decimal point is mandatory</li>
 * <li>
 * There MUST be a single non zero digit on the left of the decimal point
 * (unless a zero is represented)</li>
 * <li>
 * There MUST be at least single digit on the right of the decimal point.</li>
 * </ul>
 * <li>
 * Exponent - Zero exponent is represented by "E0"</li> <li>
 * "+" sign is prohibited in both the mantissa and the exponent.</li> <li>
 * Leading zeroes are prohibited from the left side of the decimal point in the
 * mantissa and from the exponent.</li> <li>
 * Special values (NaN, INF) MUST not be used</li> </ul>
 */
public final class JsonCanonicalizer implements Canonicalizer {

    private static final Message CANONICALIZATION_ERROR = MessageFactory
            .getMessage("vapi.signature.canonicalization");

    @Override
    public String asCanonicalString(String message) {
        Validate.notNull(message);

        ObjectMapper m = new ObjectMapper();
        SimpleModule module =
                new SimpleModule("Custom serializer", new Version(1, 0, 0, null, null, null));
        module.addSerializer(new DoubleSerializer());
        m.registerModule(module);
        JsonNode root = parseToTree(message, m);

        Object canonicalizedForm = root;
        if (root.isObject()) {
            canonicalizedForm = canonicalizeObject(root);
        } else if (root.isArray()) {
            canonicalizedForm = canonicalizeArray(root);
        }

        return serializeToString(m, canonicalizedForm);
    }

    /**
     * Canonicalizes a JSON object following the JSON canonicalization rules
     *
     * @param jsonObject
     *            not null
     * @return a tree map that contains the sorted object
     */
    private TreeMap<String, Object> canonicalizeObject(JsonNode jsonObject) {
        assert jsonObject != null;

        // TODO define comparator for strings which sorts by UCS codepoint
        // values
        TreeMap<String, Object> sorted = new TreeMap<>();
        Iterator<Entry<String, JsonNode>> fieldsIterator = jsonObject.fields();
        while (fieldsIterator.hasNext()) {
            Entry<String, JsonNode> nextField = fieldsIterator.next();
            String key = nextField.getKey();
            JsonNode value = nextField.getValue();
            if (value.isArray()) {
                sorted.put(key, canonicalizeArray(value));
            } else if (value.isObject()) {
                sorted.put(key, canonicalizeObject(value));
            } else {
                sorted.put(key, value);
            }
        }

        return sorted;
    }

    /**
     * Canonicalizes a JSON array following the JSON canonicalization rules
     *
     * @param jsonArray not null
     * @return a list containing the sorted array
     */
    private List<Object> canonicalizeArray(JsonNode jsonArray) {
        assert jsonArray != null;

        List<Object> sorted = new ArrayList<>();
        for (JsonNode element : jsonArray) {
            if (element.isArray()) {
                sorted.add(canonicalizeArray(element));
            } else if (element.isObject()) {
                sorted.add(canonicalizeObject(element));
            } else {
                sorted.add(element);
            }
        }

        return sorted;
    }

    /**
     * Serializes the canonical form into a String
     *
     * @param mapper
     *            the JSON object mapper. cannot be null
     * @param canonicalizedForm
     *            the canonical form. cannot be null
     * @return the serialized canonical form. cannot be null
     */
    private String serializeToString(ObjectMapper mapper,
                                     Object canonicalizedForm) {
        assert mapper != null && canonicalizedForm != null;

        String result;
        try {
            result = mapper.writeValueAsString(canonicalizedForm);
        } catch (JsonGenerationException e) {
            throw new SignatureException(CANONICALIZATION_ERROR, e);
        } catch (JsonMappingException e) {
            throw new SignatureException(CANONICALIZATION_ERROR, e);
        } catch (IOException e) {
            throw new SignatureException(CANONICALIZATION_ERROR, e);
        }
        return result;
    }

    /**
     * Parses the given JSON message into a tree
     *
     * @param message
     *            cannot be null
     * @param mapper
     *            cannot be null
     * @return the root node of the JSON tree
     */
    private JsonNode parseToTree(String message, ObjectMapper mapper) {
        assert message != null && mapper != null;

        JsonNode root;
        try {
            root = mapper.readTree(message);
        } catch (IOException e) {
            throw new SignatureException(CANONICALIZATION_ERROR, e);
        }
        return root;
    }

    /**
     * This class is responsible for serializing double into its canonical form.
     */
    @SuppressWarnings("serial")
    private static final class DoubleSerializer extends StdSerializer<DoubleNode> {

        protected DoubleSerializer() {
            super(DoubleNode.class);
        }

        @Override
        public void serialize(DoubleNode value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            double doubleValue = value.asDouble();
            jgen.writeNumber(
                    CanonicalizationUtil.canonicalizeDouble(doubleValue));
        }
    }
}
