/* **********************************************************
 * Copyright 2012 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.dsig.json;

import java.io.FileInputStream;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStore.PrivateKeyEntry;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.CoreException;

/**
 * Helper class: wraps a key store and provides access to the entries
 */
public class KeyStoreHelper {

    private static final String ENTRY_NOT_ACCESSIBLE = "vapi.keystore.entrynotaccessible";
    private static final String JKS_TYPE = "JKS";
    private static final Logger log =
            LoggerFactory.getLogger(KeyStoreHelper.class);
    private final KeyStore _keyStore;

    /**
     * Create a key store helper object.
     *
     * @param storeName
     *            The name of the key store.
     * @param storePass
     *            The password for the keystore.
     *
     * @throws IllegalArgumentException
     *             if the key store with the given name does not exist or cannot
     *             be loaded.
     */
    public KeyStoreHelper(String storeName, char[] storePass) {

        try {
            _keyStore = KeyStore.getInstance(JKS_TYPE);
            FileInputStream keystoreFileStream = new FileInputStream(storeName);
            try {
                _keyStore.load(keystoreFileStream, storePass);
            } finally {
                keystoreFileStream.close();
            }

        } catch (IOException e) {
            throw new IllegalArgumentException("Failed to load keystore "
                    + storeName, e);

        } catch (GeneralSecurityException e) {
            throw new IllegalArgumentException("Failed to load keystore "
                    + storeName, e);
        }
    }

    /**
     * Returns a X.509 certificate entry.
     *
     * @throws IllegalArgumentException
     *             if the entry doesn't exist
     */
    public X509Certificate getCertificate(String certAlias) {
        X509Certificate cert;
        try {
            cert = (X509Certificate) _keyStore.getCertificate(certAlias);

        } catch (ClassCastException e) {
            throw new IllegalArgumentException("Entry " + certAlias
                    + " does not correspond to an X.509 certificate", e);

        } catch (GeneralSecurityException e) {
            log.debug(e.getMessage(), e);
            throw new CoreException(ENTRY_NOT_ACCESSIBLE, certAlias);
        }

        checkEntryAvailable(certAlias, cert);

        return cert;
    }

    /**
     * Returns a PrivateKey entry.
     *
     * @throws IllegalArgumentException
     *             if the entry doesn't exist
     */
    public PrivateKey getPrivateKey(String certAlias, char[] pass) {
        KeyStore.PrivateKeyEntry pk;
        try {
            /**
             * Helper: check that the object returned for the given entry is a
             * real thing (because the KeyStore won't do it for us).
             */

            pk = (PrivateKeyEntry) _keyStore.getEntry(certAlias,
                    new KeyStore.PasswordProtection(pass));

        } catch (ClassCastException e) {
            throw new IllegalArgumentException("Entry " + certAlias
                    + " does not correspond to a private key", e);

        } catch (GeneralSecurityException e) {
            log.debug(e.getMessage(), e);
            throw new CoreException(ENTRY_NOT_ACCESSIBLE, certAlias);
        }

        checkEntryAvailable(certAlias, pk);

        return pk.getPrivateKey();
    }

    /**
     * Helper: check that the object returned for the given entry is a real
     * thing (because the KeyStore won't do it for us).
     */
    private void checkEntryAvailable(String certAlias, Object ksObj) {
        if (ksObj == null) {
            throw new IllegalArgumentException("Entry " + certAlias
                    + " not found in store");
        }
    }
}
