/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol;

import java.util.concurrent.Executor;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.DecoratorApiProvider;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.internal.util.Validate;

/**
 * Decorator which allows to run ApiProvider methods asynchronously.
 */
public final class ExecutorApiProvider extends DecoratorApiProvider {
    private final Executor executor;

    /**
     * Creates a decorator around an existing ApiProvider.
     *
     * @param apiProvider the ApiProvider; must not be <code>null</code>
     * @param executor executor used to execute the provider methods;
     *                 must not be <code>null</code>
     */
    public ExecutorApiProvider(ApiProvider apiProvider,
                               Executor executor) {
        super(apiProvider);
        Validate.notNull(executor);
        this.executor = executor;
    }

    @Override
    public void invoke(final String serviceId,
                       final String operationId,
                       final DataValue input,
                       final ExecutionContext ctx,
                       final AsyncHandle<MethodResult> asyncHandle) {
        Runnable task = new Runnable() {
            @Override
            public void run() {
                try {
                    decoratedProvider.invoke(serviceId,
                                             operationId,
                                             input,
                                             ctx,
                                             asyncHandle);
                } catch (RuntimeException e) {
                    asyncHandle.setError(e);
                }
            }
        };
        executor.execute(task);
    }
}
