/* **********************************************************
 * Copyright (c) 2016-2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rest;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vmware.vapi.internal.bindings.OperationDef;
import com.vmware.vapi.internal.util.Validate;

/**
 * Provider for REST metadata which is fed with static bindings generated
 * metadata.
 *
 * <p>An instance of this class is supposed to be initialized with the metadata
 * for all operations of a given API service. So a separate instance is needed
 * for each API service (and respectively bindings stub).
 *
 * <p>The above means that the {@code serviceId} parameter of the lookup
 * method {@link #getRestMetadata} is not really used to find the
 * metadata for the proper operation.
 */
public class BindingsOperationRestMetadataProvider implements OperationRestMetadataProvider {
    // key is operationId
    private final Map<String, BindingsOperationRestMetadata> metadataPerOp =
            new HashMap<>();

    public BindingsOperationRestMetadataProvider(List<OperationDef> operationDefs) {
        Validate.notNull(operationDefs);
        for (OperationDef oper : operationDefs) {
            metadataPerOp.put(oper.getOperationId(),
                              new BindingsOperationRestMetadata(oper));
        }
    }

    /**
     * @param serviceId not used
     */
    @Override
    public OperationRestMetadata getRestMetadata(String serviceId,
                                                 String operationId) {
        return metadataPerOp.get(operationId);
    }
}
