/* **********************************************************
 * Copyright (c) 2017-2018, 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rest;

import java.util.Objects;

import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.internal.protocol.client.rpc.HttpRequest;
import com.vmware.vapi.internal.protocol.client.rpc.HttpRequest.HttpResponseHandler;
import com.vmware.vapi.internal.protocol.client.rpc.HttpResponse;
import com.vmware.vapi.internal.protocol.client.rpc.RestTransport;
import com.vmware.vapi.internal.protocol.common.msg.JsonMessageProtocolExceptionTranslator;

/**
 * The default implementation of {@link RequestExecutorFactory}. It creates
 * {@link RequestExecutor} instances which do nothing more than execute the
 * specified {@link HttpRequest}, parse the result and report it.
 */
public class DefaultRequestExecutorFactory implements RequestExecutorFactory {
    @Override
    public RequestExecutor createRequestExecutor(RestTransport transport,
                                                 ResponseParser parser) {
        return new DefaultRequestExecutor(transport, parser);
    }

    private static class DefaultRequestExecutor implements RequestExecutor {
        private final RestTransport transport;
        private final ResponseParser parser;

        DefaultRequestExecutor(RestTransport transport, ResponseParser parser) {
            this.transport = transport;
            this.parser = parser;
        }

        @Override
        public void execute(final String serviceId,
                            final String operationId,
                            final HttpRequest request,
                            final ExecutionContext ctx,
                            final AsyncHandle<MethodResult> asyncHandle) {
            transport.execute(Objects.requireNonNull(request),
                              new DefaultHttpResponseHandler(parser,
                                                             serviceId,
                                                             operationId,
                                                             asyncHandle),
                              ctx);
        }
    }

    /**
     * The default implementation of {@link HttpResponseHandler}. It parses the
     * http-response and passes on the result to the next layer, one step closer
     * to the consumer.
     *
     */
    public static class DefaultHttpResponseHandler
            implements HttpResponseHandler {

        protected final ResponseParser parser;
        protected final String serviceId;
        protected final String operationId;
        protected final AsyncHandle<MethodResult> asyncHandle;

        /**
         * @param parser to parse the HTTP result; must not be {@code null}
         * @param serviceId service identifier; must not be {@code null}
         * @param operationId operation identifier; must not be {@code null}
         * @param asyncHandle handle used to asynchronously "return" the result
         *        or error of the invocation; must not be {@code null}
         */
        public DefaultHttpResponseHandler(ResponseParser parser,
                                          String serviceId,
                                          String operationId,
                                          AsyncHandle<MethodResult> asyncHandle) {
            this.parser = Objects.requireNonNull(parser);
            this.serviceId = Objects.requireNonNull(serviceId);
            this.operationId = Objects.requireNonNull(operationId);
            this.asyncHandle = Objects.requireNonNull(asyncHandle);
        }

        @Override
        public void onResult(HttpResponse response) {
            asyncHandle.setResult(parser.parse(serviceId,
                                               operationId,
                                               response));

        }

        @Override
        public void onError(Exception ex) {
            asyncHandle.setError(JsonMessageProtocolExceptionTranslator.translate(ex));
        }

    }
}
