/* **********************************************************
 * Copyright (c) 2015-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rest;

import java.util.List;
import java.util.Map.Entry;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.RuntimeData;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.protocol.client.rpc.HttpRequest;
import com.vmware.vapi.internal.protocol.client.rpc.http.HttpRequestImpl;
import com.vmware.vapi.internal.protocol.client.rpc.http.ProcessorManager;
import com.vmware.vapi.internal.protocol.common.msg.JsonMessageProtocolExceptionTranslator;
import com.vmware.vapi.internal.util.Validate;

// TODO add unit tests, logging and exception handling. It must be thread safe!
/**
 * {@link ApiProvider} client-side implementation which invokes
 * remote APIs using REST protocol.
 */
public class RestClientApiProvider extends ProcessorManager implements ApiProvider {
    private RequestExecutor requestExecutor;
    private RequestBuilderFactory requestBuilderFactory;

    /**
     * Constructor.
     *
     * @param requestExecutor responsible for executing the HTTP requests; must
     *        not be {@code null}
     * @param requestBuilderFactory factory for {@link RequestBuilder}s which
     *        are responsible for building proper REST request; must not be
     *        {@code null}
     */
    public RestClientApiProvider(RequestExecutor requestExecutor,
                                 RequestBuilderFactory requestBuilderFactory) {
        Validate.notNull(requestExecutor);
        Validate.notNull(requestBuilderFactory);
        this.requestExecutor = requestExecutor;
        this.requestBuilderFactory = requestBuilderFactory;
    }

    @Override
    public void invoke(String serviceId,
                       String operationId,
                       DataValue input,
                       ExecutionContext ctx,
                       AsyncHandle<MethodResult> asyncHandle) {
        Validate.notNull(serviceId);
        Validate.notNull(operationId);
        Validate.notNull(input);
        Validate.notNull(ctx);
        Validate.notNull(asyncHandle);

        try {
            RequestBuilder reqBuilder = requestBuilderFactory
                    .createRequestBuilder(serviceId, operationId);
            StructValue inputStruct = (StructValue) input;
            HttpRequest request = buildHttpRequest(reqBuilder, inputStruct, ctx);
            runRequestProcessors(serviceId, operationId, request, inputStruct, ctx);
            requestExecutor.execute(serviceId, operationId, request, ctx, asyncHandle);
        } catch (RuntimeException exception) {
            asyncHandle.setError(JsonMessageProtocolExceptionTranslator.translate(exception));
        }
    }

    private HttpRequest buildHttpRequest(RequestBuilder requestBuilder,
                                         StructValue inputStruct,
                                         ExecutionContext ctx) {
        HttpRequestImpl request = new HttpRequestImpl();
        request.setUrl(requestBuilder.buildUrl(inputStruct));
        request.setBody(requestBuilder.buildBody(inputStruct));
        request.setMethod(requestBuilder.getMethod());

        if (ctx != null) {
            RuntimeData runtimeData = ctx.retrieveRuntimeData();
            if (runtimeData != null) {
                request.setReadTimeout(runtimeData.getReadTimeout());
            }
        }

        for (Entry<String, List<String>> headers : requestBuilder.buildHeaders(inputStruct).entrySet()) {
            String headerName = headers.getKey();
            for (String headerValue : headers.getValue()) {
                request.addHeader(headerName, headerValue);
            }
        }
        return request;
    }

    private void runRequestProcessors(String serviceId,
                                      String operationId,
                                      HttpRequest request,
                                      StructValue inputStruct,
                                      ExecutionContext ctx) {
        this.handle(serviceId, operationId, request, inputStruct, ctx);
    }
}
