/* **********************************************************
 * Copyright (c) 2016-2017, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rest;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import com.vmware.vapi.client.exception.ClientException;
import com.vmware.vapi.client.exception.MessageProtocolException;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.protocol.common.DirectSerializer;
import com.vmware.vapi.internal.protocol.common.SerializerConfig;
import com.vmware.vapi.internal.protocol.common.http.UrlUtil;
import com.vmware.vapi.internal.util.Validate;

/**
 * Factory for creating {@link RequestBuilder}s for <b>REST Native</b> invocations.
 *
 * <p>This factory is using the metadata-free {@link DirectSerializer serializer}
 * to produce the request body.
 */
public class RestRequestBuilderFactory implements RequestBuilderFactory {
    private final DirectSerializer serializer;
    private final OperationRestMetadataProvider restMetadataProvider;
    private final UrlUtil urlUtil;
    protected final String baseUrl;

    public RestRequestBuilderFactory(String baseUrl,
                                     DirectSerializer serializer,
                                     OperationRestMetadataProvider restMetadataProvider,
                                     UrlUtil urlUtil) {
        Validate.notNull(baseUrl);
        Validate.notNull(serializer);
        Validate.notNull(restMetadataProvider);
        Validate.notNull(urlUtil);
        this.baseUrl = baseUrl;
        this.serializer = serializer;
        this.restMetadataProvider = restMetadataProvider;
        this.urlUtil = urlUtil;
    }

    @Override
    public RequestBuilder createRequestBuilder(String serviceId, String operationId) {
        OperationRestMetadata restMetadata =
                restMetadataProvider.getRestMetadata(serviceId, operationId);

        if (restMetadata == null) {
            throw new ClientException("Missing REST metadata in generated bindings.");
        }

        return new RestRequestBuilder(restMetadata,
                                      new RestMetadataBasedHttpRequestBuilder(restMetadata,
                                                                              urlUtil));
    }

    private final class RestRequestBuilder extends DefaultRequestBuilder {
        public RestRequestBuilder(OperationRestMetadata operationMetadata,
                                RestMetadataBasedHttpRequestBuilder httpRequestBuilder) {
            super(httpRequestBuilder, operationMetadata, baseUrl);
        }

        @Override
        public byte[] buildBody(StructValue input) {
            DataValue requestBodyValue =
                    getHttpRequestBuilder().getRequestBodyValue(input);
            if (requestBodyValue == null) {
                return null;
            }

            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            SerializerConfig writerConfig = new SerializerConfig(false, false);
            try {
                serializer.serialize(requestBodyValue, outputStream, writerConfig);
                return outputStream.toByteArray();
            } catch (IOException e) {
                throw new MessageProtocolException("Request body serialization failed.", e);
            }
        }
    }
}
