/* **********************************************************
 * Copyright (c) 2015-2016, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rest;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.client.exception.InternalClientException;
import com.vmware.vapi.client.exception.TransportProtocolException;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.data.VoidValue;
import com.vmware.vapi.internal.protocol.client.rpc.HttpResponse;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.std.StandardDataFactory;
import com.vmware.vapi.std.errors.ConcurrentChange;
import com.vmware.vapi.std.errors.Error;
import com.vmware.vapi.std.errors.InternalServerError;
import com.vmware.vapi.std.errors.InvalidRequest;
import com.vmware.vapi.std.errors.NotFound;
import com.vmware.vapi.std.errors.ResourceBusy;
import com.vmware.vapi.std.errors.ResourceInaccessible;
import com.vmware.vapi.std.errors.ServiceUnavailable;
import com.vmware.vapi.std.errors.TimedOut;
import com.vmware.vapi.std.errors.UnableToAllocateResource;
import com.vmware.vapi.std.errors.Unauthenticated;
import com.vmware.vapi.std.errors.Unauthorized;

/**
 * Error converter that convert REST errors to vAPI errors using only the
 * response status code.
 */
public class StatusCodeErrorConverter implements ErrorConverter {
    private static final Logger LOGGER = LoggerFactory.getLogger(StatusCodeErrorConverter.class);

    private static final Map<Integer, Class<? extends Error>> ERROR_MAP =
            new HashMap<>();

    static {
        ERROR_MAP.put(400, InvalidRequest.class);
        ERROR_MAP.put(401, Unauthenticated.class);
        ERROR_MAP.put(402, Unauthorized.class);
        ERROR_MAP.put(403, Unauthorized.class);
        ERROR_MAP.put(404, NotFound.class);
        ERROR_MAP.put(405, InvalidRequest.class);
        ERROR_MAP.put(406, InvalidRequest.class);
        ERROR_MAP.put(407, Unauthenticated.class);
        ERROR_MAP.put(408, TimedOut.class);
        ERROR_MAP.put(409, ConcurrentChange.class);
        ERROR_MAP.put(410, NotFound.class);
        ERROR_MAP.put(411, InvalidRequest.class);
        ERROR_MAP.put(412, InvalidRequest.class);
        ERROR_MAP.put(413, InvalidRequest.class);
        ERROR_MAP.put(414, InvalidRequest.class);
        ERROR_MAP.put(415, InvalidRequest.class);
        ERROR_MAP.put(416, ResourceInaccessible.class);
        ERROR_MAP.put(417, InvalidRequest.class);
        ERROR_MAP.put(418, Error.class);
        ERROR_MAP.put(422, InvalidRequest.class);
        ERROR_MAP.put(423, ResourceBusy.class);
        ERROR_MAP.put(424, InvalidRequest.class);
        ERROR_MAP.put(426, InvalidRequest.class);
        ERROR_MAP.put(428, ConcurrentChange.class);
        ERROR_MAP.put(429, ServiceUnavailable.class);
        ERROR_MAP.put(431, InvalidRequest.class);
        ERROR_MAP.put(500, InternalServerError.class);
        ERROR_MAP.put(501, Error.class);
        ERROR_MAP.put(502, InternalServerError.class);
        ERROR_MAP.put(503, ServiceUnavailable.class);
        ERROR_MAP.put(504, TimedOut.class);
        ERROR_MAP.put(505, InvalidRequest.class);
        ERROR_MAP.put(506, InternalServerError.class);
        ERROR_MAP.put(507, UnableToAllocateResource.class);
        ERROR_MAP.put(508, InternalServerError.class);
        ERROR_MAP.put(509, UnableToAllocateResource.class);
        ERROR_MAP.put(510, InvalidRequest.class);
        ERROR_MAP.put(511, Unauthenticated.class);
    }

    private final BodyConverter bodyConverter;

    public StatusCodeErrorConverter(BodyConverter bodyConverter) {
        Validate.notNull(bodyConverter);
        this.bodyConverter = bodyConverter;
    }

    @Override
    public ErrorValue getError(HttpResponse response) {
        // TODO: rest native: dependency on the bindings layer, shouldn't
        //       really exists here; instead use runtime mechanisms to build
        //       ErrorValue matching the std errors
        //       Also Error limits support to std Error; probably OK for now
        //       but could use ApiError instead
        Class<? extends Error> bindingClass = ERROR_MAP.get(response.getStatusCode());
        if (bindingClass == null) {
            throw new TransportProtocolException("Unknown status code " + response.getStatusCode());
        }

        return buildErrorValue(response, bindingClass);
    }

    private ErrorValue buildErrorValue(HttpResponse response,
                                       Class<? extends Error> bindingClass) {
        ErrorValue result = buildEmptyErrorValue(bindingClass);
        DataValue errorData = parseResponseBody(response);
        if (errorData != VoidValue.getInstance()) {
            result.setField(StandardDataFactory.DATA_FIELD_NAME, errorData);
        }
        return result;
    }

    private ErrorValue buildEmptyErrorValue(Class<? extends Error> bindingClass) {
        try {
            // CCE is desired if it is not
            return (ErrorValue) bindingClass.newInstance()._getDataValue();
        } catch (InstantiationException | IllegalAccessException ex) {
            LOGGER.error("Failed to instantiate binding class for vAPI error", ex);
            throw new InternalClientException(ex);
        }
    }

    private DataValue parseResponseBody(HttpResponse response) {
        return bodyConverter.getResult(null, null, response);
    }
}