/* **********************************************************
 * Copyright 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc.http;

import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.CancellationException;

import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.protocol.HttpContext;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.internal.protocol.client.rpc.HttpRequest;
import com.vmware.vapi.internal.protocol.client.rpc.HttpRequest.HttpResponseHandler;
import com.vmware.vapi.internal.protocol.client.rpc.RestTransport;
import com.vmware.vapi.internal.protocol.client.rpc.http.ConnectionMonitor.CleanableConnectionPool;
import com.vmware.vapi.internal.protocol.common.http.ApacheHttpClientExceptionTranslator;

/**
 * WARNING: This class could be used by vAPI Endpoint or API Gateway.
 */
public class ApacheAsyncClientRestTransport implements RestTransport {

    private final CloseableHttpAsyncClient httpClient;
    private final ApacheClientRequestConfigurationMerger configMerger;

    // we need a hard reference to this for preemptive connection clean-up to work
    @SuppressWarnings("unused")
    private final CleanableConnectionPool connectionPool;

    public ApacheAsyncClientRestTransport(CloseableHttpAsyncClient httpClient,
                                          RequestConfig defaultRequestConfig) {
        this(httpClient, defaultRequestConfig, null);
    }

    public ApacheAsyncClientRestTransport(CloseableHttpAsyncClient httpClient,
                                          RequestConfig defaultRequestConfig,
                                          CleanableConnectionPool connectionPool) {
        this.httpClient = Objects.requireNonNull(httpClient);
        this.configMerger = new ApacheClientRequestConfigurationMerger(defaultRequestConfig);
        this.connectionPool = connectionPool;
    }

    @Override
    public void execute(HttpRequest request,
                        HttpResponseHandler responseHandler,
                        ExecutionContext ctx) {
        Objects.requireNonNull(request);
        Objects.requireNonNull(request.getUrl());
        Objects.requireNonNull(request.getMethod());
        Objects.requireNonNull(responseHandler);
        Objects.requireNonNull(ctx);
        HttpUriRequest apacheRequest = ApacheClientRestTransport.createRequest(request);
        ApacheClientRestTransport.addRequestHeaders(request, apacheRequest);
        ApacheClientRestTransport.addRequestBody(request, apacheRequest);
        HttpContext context = ApacheHttpUtil.createHttpContext(request.getReadTimeout(), configMerger);
        httpClient.execute(apacheRequest, context, new ResponseCallback(responseHandler, ctx));
    }

    @Override
    public void close() throws IOException {
        httpClient.close();
    }

    private static final class ResponseCallback implements FutureCallback<org.apache.http.HttpResponse> {

        private final HttpResponseHandler responseHandler;
        private final ExecutionContext ctx;

        private ResponseCallback(HttpResponseHandler responseHandler, ExecutionContext ctx) {
            this.responseHandler = responseHandler;
            this.ctx = ctx;
        }

        @Override
        public void completed(org.apache.http.HttpResponse apacheResponse) {
            ApacheHttpResponse response = new ApacheHttpResponse(apacheResponse);
            ApacheClientRestTransport.handleResponseAccessors(response, ctx);

            try {
                responseHandler.onResult(response);
            } catch (RuntimeException e) {
                this.failed(e);
            }
        }

        @Override
        public void failed(Exception ex) {
            responseHandler.onError(ApacheHttpClientExceptionTranslator.translate(ex, null));
        }

        @Override
        public void cancelled() {
            failed(new CancellationException());
        }
    }
}
