/* **********************************************************
 * Copyright (c) 2015-2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http;

import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.client.LaxRedirectStrategy;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.protocol.client.rpc.http.ConnectionMonitor.CleanableConnectionPool;
import com.vmware.vapi.internal.protocol.client.rpc.http.adapter.BioHttpClientBuilderAdapter;
import com.vmware.vapi.protocol.HttpConfiguration;
import com.vmware.vapi.protocol.HttpConfiguration.HeadersProvider;
import com.vmware.vapi.protocol.HttpConfiguration.SslConfiguration;

/**
 * Builder for Apache HTTP BIO Clients.
 */

public class ApacheBioHttpClientBuilder {
    final static String HTTP_SCHEME = "http";
    final static String HTTPS_SCHEME = "https";

    private PoolingHttpClientConnectionManager connManager;
    private RequestConfig defaultRequestConfig;

    private static Logger logger = LoggerFactory.getLogger(ApacheBioHttpClientBuilder.class);
    private CleanableConnectionPool pool;

    /**
     * @return the default configuration the HTTP client was built with;
     *         {@code null} if it has not been built yet
     */
    public RequestConfig getDefaultRequestConfig() {
       return defaultRequestConfig;
    }

    /**
     * @return the connection manager used to build the HTTP client;
     *             {@code null} if it has not been built yet
     */
    public HttpClientConnectionManager getConnectionManager() {
        return connManager;
    }

    /**
     * Creates apache HTTP client instance and configures it accordingly to the specified
     * {@code HttpConfiguration}.
     *
     * <p>It is responsibility of the consuming code to properly dispose the client when it is no
     * longer needed.
     *
     * @param httpConfig http settings to be applied
     *
     * @return closable http client
     *
     * @throws SecurityException
     * If a security manager has been installed and it denies
     * {@link java.net.NetPermission}{@code ("getProxySelector")}
     * when using system default proxy configuration
     * since {@link java.net.ProxySelector#getDefault()} is used in this case.
     */
    public CloseableHttpClient buildAndConfigure(final HttpConfiguration httpConfig) throws SecurityException {
        connManager = createConnectionManager(httpConfig);
        defaultRequestConfig = ApacheHttpUtil.createDefaultRequestConfig(httpConfig);

        HttpClientBuilder builder = HttpClients.custom()
           .setConnectionManager(connManager)
           .setUserAgent(ApacheHttpUtil.VAPI_USER_AGENT)
           .setDefaultRequestConfig(defaultRequestConfig);
        // TODO: reconsider replacing the strategy usage here with just
        //       setting the limits to the ctor of the connection pool
        //       we don't really need the strategy since we don't base our
        //       decision here on the actual last response (headers)
        //       OR
        //       maybe we need more sophisticated protocol to communicate
        //       the time to live using the keep-alive HTTP header (which we
        //       don't use today)
        builder.setKeepAliveStrategy(new ConnectionKeepAliveStrategy() {
            @Override
            public long getKeepAliveDuration(HttpResponse response,
                                             HttpContext context) {
                return httpConfig.getKeepAlivePeriod();
            }
        });
        //in addition to GET/HEAD follow redirects for POST/DELETE methods as well: RFC7231 section 6.4.7
        builder.setRedirectStrategy(new LaxRedirectStrategy());
        HeadersProvider headersProvider = httpConfig.getHeadersProvider();
        if (headersProvider != null) {
            builder.addInterceptorFirst(
                         new ApacheClientHeadersProvider(headersProvider));
        }

        HttpConfiguration.ProxyConfiguration proxyConfig = httpConfig.getProxyConfiguration();
        if (proxyConfig != null) {
            HttpClientProxyConfigurationHelper.addProxyConfiguration(
                proxyConfig,
                new BioHttpClientBuilderAdapter(builder));
        }

        return builder.build();
    }

    /**
     * Registers the connection pool manager only once with a
     * {@code ConnectionMonitor} thread utilized for cleaning stale connections.
     * The returned object needs to be kept as a hard link, in order for
     * successful monitoring to be accomplished.
     *
     * @return CleanableConnectionPool a strong reference pointer
     */
    public CleanableConnectionPool registerClientWithConnectionMonitor() {
        if (connManager == null) {
            throw new IllegalStateException();
        }

        if (pool == null) {
            pool = new CleanableConnectionPool() {
                @Override
                public void closeExpiredConnections() {
                    connManager.closeExpiredConnections();
                }

                @Override
                public int leasedConnections() {
                    return connManager.getTotalStats().getLeased();
                }
            };

            ConnectionMonitor.register(pool);
        } else {
            logger.warn("Unable to register client more than once!");
        }

        return pool;
    }

    private PoolingHttpClientConnectionManager createConnectionManager(
            HttpConfiguration httpConfig) {
        Registry<ConnectionSocketFactory> registry =
                createConnectionSocketFactoryRegistry(httpConfig.getSslConfiguration());
        PoolingHttpClientConnectionManager result =
                new PoolingHttpClientConnectionManager(registry);
        result.setMaxTotal(httpConfig.getMaxConnections());
        result.setDefaultMaxPerRoute(httpConfig.getMaxConnections());
        return result;
    }

    @SuppressWarnings("deprecation")
    Registry<ConnectionSocketFactory> createConnectionSocketFactoryRegistry(
            SslConfiguration sslConfig) {
        RegistryBuilder<ConnectionSocketFactory> registryBuilder =
                RegistryBuilder.<ConnectionSocketFactory>create()
                               .register(HTTP_SCHEME, PlainConnectionSocketFactory.INSTANCE);

        if (sslConfig != SslConfiguration.SKIP_SSL_INITIALIZATION) {
            org.apache.http.conn.ssl.X509HostnameVerifier hostnameVerifier =
                    SslClientUtil.createHostnameVerifier(
                               sslConfig.isHostnameVerificationDisabled(),
                               SSLConnectionSocketFactory.STRICT_HOSTNAME_VERIFIER);

            registryBuilder.register(HTTPS_SCHEME,
                                     new SSLConnectionSocketFactory(
                                             SslClientUtil.createSslContext(sslConfig),
                                             sslConfig.getEnabledProtocols(),
                                             sslConfig.getEnabledCipherSuites(),
                                             hostnameVerifier));
        }
        return registryBuilder.build();
    }
}
