/* **********************************************************
 * Copyright (c) 2015-2016, 2018-2020, 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http;


import java.io.IOException;
import java.util.List;
import java.util.Map.Entry;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpOptions;
import org.apache.http.client.methods.HttpPatch;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHeader;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.RuntimeData;
import com.vmware.vapi.internal.protocol.client.rpc.HttpRequest;
import com.vmware.vapi.internal.protocol.client.rpc.HttpRequest.HttpResponseHandler;
import com.vmware.vapi.internal.protocol.client.rpc.RestTransport;
import com.vmware.vapi.internal.protocol.client.rpc.http.ConnectionMonitor.CleanableConnectionPool;
import com.vmware.vapi.internal.protocol.common.http.ApacheHttpClientExceptionTranslator;
import com.vmware.vapi.internal.util.Validate;

public class ApacheClientRestTransport implements RestTransport {
    private static final Logger logger =
            LoggerFactory.getLogger(ApacheClientRestTransport.class);

    private final CloseableHttpClient httpClient;
    private final ApacheClientRequestConfigurationMerger configMerger;

    // we need a hard reference to this for preemptive connection clean-up to work
    @SuppressWarnings("unused")
    private final CleanableConnectionPool connectionPool;

    public ApacheClientRestTransport(CloseableHttpClient httpClient) {
       this(httpClient, null);
    }

    public ApacheClientRestTransport(CloseableHttpClient httpClient,
                                     RequestConfig defaultRequestConfig) {
        this(httpClient, defaultRequestConfig, null);
    }

    public ApacheClientRestTransport(CloseableHttpClient httpClient,
                                     RequestConfig defaultRequestConfig,
                                     CleanableConnectionPool connectionPool) {
        // TODO: in this ctor it might not be this class repsonsibility
        //       to close the client, whicle in the overload it clearly is
        //       make sure the contract is clear and well documented
        this.httpClient = httpClient;
        this.configMerger =
                new ApacheClientRequestConfigurationMerger(defaultRequestConfig);
        this.connectionPool = connectionPool;
    }

    @Override
    public void execute(HttpRequest request,
                        HttpResponseHandler responseHandler,
                        ExecutionContext ctx) {
        // TODO: the notNulls
        Validate.notNull(request.getUrl(), "Url of request is null. Cannot execute it.");
        Validate.notNull(request.getMethod(), "Method of request is null. Cannot execute it.");
        HttpUriRequest apacheRequest = createRequest(request);
        addRequestHeaders(request, apacheRequest);
        addRequestBody(request, apacheRequest);

        HttpContext context =
                ApacheHttpUtil.createHttpContext(request.getReadTimeout(),
                                                 configMerger);
        try {
            org.apache.http.HttpResponse apacheResponse =
                    httpClient.execute(apacheRequest, context);
            ApacheHttpResponse response = new ApacheHttpResponse(apacheResponse);
            handleResponseAccessors(response, ctx);
            responseHandler.onResult(response);
        } catch (Exception e) {
            responseHandler.onError(ApacheHttpClientExceptionTranslator.translate(e, null));
        }
    }

    @Override
    public void close() throws IOException {
        httpClient.close();
    }

    static HttpUriRequest createRequest(HttpRequest request) {
        switch (request.getMethod()) {
            case GET:
                return new HttpGet(request.getUrl());
            case HEAD:
                return new HttpHead(request.getUrl());
            case POST:
                return new HttpPost(request.getUrl());
            case PUT:
                return new HttpPut(request.getUrl());
            case DELETE:
                return new HttpDelete(request.getUrl());
            case OPTIONS:
                return new HttpOptions(request.getUrl());
            case PATCH:
                return new HttpPatch(request.getUrl());

            default:
                IllegalArgumentException e = new IllegalArgumentException(
                        "Unsupported HTTP method requested: "+ request.getMethod());
                logger.error(e.getMessage(), e);
                throw e;
        }
    }

    static void addRequestHeaders(HttpRequest request, HttpUriRequest apacheRequest) {
        for (Entry<String, List<String>> headersForName : request.getHeaders().entrySet()) {
            for (String value : headersForName.getValue()) {
                Header header = new BasicHeader(headersForName.getKey(), value);
                apacheRequest.addHeader(header);
            }
        }
    }

    static void addRequestBody(HttpRequest request, HttpUriRequest apacheRequest) {
        // Only POST and PUT are HttpEntityEnclosingRequest and they require
        // body. So - create and set the entity representing the body.
        if (apacheRequest instanceof HttpEntityEnclosingRequest && request.getBody() != null) {
            HttpEntityEnclosingRequest entityEnclosingRequest =
                    (HttpEntityEnclosingRequest) apacheRequest;
            HttpEntity requestEntity = new ByteArrayEntity(request.getBody());
            entityEnclosingRequest.setEntity(requestEntity);
        }
    }

    static void handleResponseAccessors(ApacheHttpResponse response,
                                         ExecutionContext ctx) {
        RuntimeData runtimeData = ctx.retrieveRuntimeData();
        if (runtimeData == null || runtimeData.getResponseAccessor() == null) {
            return;
        }

        try {
            runtimeData.getResponseAccessor().access(response);
        } catch (RuntimeException ex) {
            logger.warn("Ignoring unexpected exception from ResponseAccessor " +
                    "(enable debug logs)");
            logger.debug("Raw response accessor exception", ex);
        }
    }
}