/* **********************************************************
 * Copyright (c) 2013-2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http;

import java.io.IOException;
import java.util.concurrent.CancellationException;

import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.nio.client.methods.HttpAsyncMethods;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.core.abort.AbortHandle;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient;
import com.vmware.vapi.internal.protocol.client.rpc.http.ConnectionMonitor.CleanableConnectionPool;
import com.vmware.vapi.internal.protocol.client.rpc.http.handle.NioMainResponseConsumer;
import com.vmware.vapi.internal.protocol.common.Util;
import com.vmware.vapi.internal.protocol.common.http.ApacheHttpClientExceptionTranslator;
import com.vmware.vapi.internal.tracing.TracingSpan;
import com.vmware.vapi.internal.tracing.otel.TracingAttributeKey.IoType;
import com.vmware.vapi.internal.util.TracingUtil;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.HttpConfiguration;
import com.vmware.vapi.protocol.HttpConfiguration.Protocol;

/**
 * <p>
 * Client transport based on Apache HttpAsyncClient.
 * </p>
 * <p>
 * This transport is truly asynchronous - the {@link #send} invocation will not block
 * while waiting for response from the server.
 * </p>
 * <p>
 * The underlying library keeps a pool of I/O threads (reactor pattern).
 * Response callbacks are invoked on the I/O threads, so callbacks should be
 * lightweight and not block for long time.
 * </p>
 */
public final class ApacheHttpAsyncClientTransport implements CorrelatingClient {

    private static Logger logger =
            LoggerFactory.getLogger(ApacheHttpAsyncClientTransport.class);

    private final String uri;
    private final CloseableHttpAsyncClient httpClient;
    private final ApacheClientRequestConfigurationMerger configMerger;
    private final Protocol protocol;
    private final int maxResponseSize;

    // Hard link to connection pool used for monitoring client connections
    private CleanableConnectionPool pool;

    /**
     * @param uri HTTP endpoint to contact; must not be <code>null</code>
     */
    public ApacheHttpAsyncClientTransport(String uri) {
        this(uri, new HttpConfiguration.Builder().getConfig());
    }


    /**
     * @param uri HTTP endpoint to contact; must not be <code>null</code>
     * @param httpConfig HTTP configuration' must not be <code>null</code>
     */
    public ApacheHttpAsyncClientTransport(String uri,
                                          HttpConfiguration httpConfig) {
        this(uri, httpConfig, new ApacheNioHttpClientBuilder());
    }


    // An intermediate construction point was introduced in order to
    // create a connection pool hard link and keep the client registered
    private ApacheHttpAsyncClientTransport(String uri,
                                          HttpConfiguration httpConfig,
                                          ApacheNioHttpClientBuilder builder) {
        this(uri,
             builder.buildAndConfigure(httpConfig),
             ApacheHttpUtil.createDefaultRequestConfig(httpConfig),
             httpConfig.getProtocol(),
             httpConfig.getMaxResponseSize());
        pool = builder.registerClientWithConnectionMonitor();
    }

    /**
     * This method creates and returns an Apache NIO client.
     * The method is not registering the client to the connection monitor,
     * hence it is the consumer's job to implement monitoring.
     *
     * @param HttpConfiguration httpConfig
     *
     * @return CloseableHttpAsyncClient Apache NIO Client
     */
    public static CloseableHttpAsyncClient createDefaultHttpClient(HttpConfiguration httpConfig) {
        ApacheNioHttpClientBuilder builder = new ApacheNioHttpClientBuilder();
        return builder.buildAndConfigure(httpConfig);
    }

    /**
     * @param uri HTTP endpoint to contact; must not be <code>null</code>
     * @param httpClient client instance; must be activated; will be shutdown
     *                   by the {@link #close()} method
     * @param executor (optional) executor; if specified, it will be used to
     *                 execute the callback methods
     */
    public ApacheHttpAsyncClientTransport(String uri,
                                          CloseableHttpAsyncClient httpClient,
                                          Protocol protocol) {
        this(uri, httpClient, null, protocol);
    }

    /**
     * @param uri HTTP endpoint to contact; must not be <code>null</code>
     * @param httpClient client instance; must be activated; will be shutdown
     *                   by the {@link #close()} method
     * @param defaultConfiguration default request configuration.
     * @param executor (optional) executor; if specified, it will be used to
     *                 execute the callback methods
     * @param protocol vAPI Protocol.
     */
    public ApacheHttpAsyncClientTransport(String uri,
                                           CloseableHttpAsyncClient httpClient,
                                           RequestConfig defaultConfiguration,
                                           Protocol protocol) {
        this(uri,
             httpClient,
             defaultConfiguration,
             protocol,
             Integer.MAX_VALUE);

    }

    /**
     * @param uri HTTP endpoint to contact; must not be <code>null</code>
     * @param httpClient client instance; must be activated; will be shutdown by
     *        the {@link #close()} method
     * @param defaultConfiguration default request configuration.
     * @param executor (optional) executor; if specified, it will be used to
     *        execute the callback methods
     * @param protocol vAPI Protocol
     * @param maxResponseSize the maximum allowed size of the content of the
     *        response that the client can consume. The size is expected to be
     *        a power of 2
     */
    public ApacheHttpAsyncClientTransport(String uri,
                                          CloseableHttpAsyncClient httpClient,
                                          RequestConfig defaultConfiguration,
                                          Protocol protocol,
                                          int maxResponseSize) {
        // TODO - validate that URI has port set
        Validate.notNull(uri);
        Validate.notNull(httpClient);
        Validate.isTrue(maxResponseSize > 0);
        this.uri = uri;
        this.protocol = protocol;
        this.httpClient = httpClient;
        this.configMerger =
                new ApacheClientRequestConfigurationMerger(defaultConfiguration);
        this.maxResponseSize = maxResponseSize;
    }

    @Override
    public void send(SendParams params) {
        final HttpPost post = new HttpPost(uri);
        post.setEntity(new InputStreamEntity(params.getRequest(), params.getRequestLength()));

        Util.addHeaders(post,
                        params.getContentType(),
                        params.getAcceptedTypes(),
                        protocol,
                        params.getServiceId(),
                        params.getOperationId(),
                        params.getExecutionContext());

        if (Util.checkRequestAborted(params.getAbortHandle(), params.getCbFactory())) {
            // Request aborted, no need to continue
            return;
        } else {
            Util.registerAbortListerner(params.getCbFactory(), post, params.getAbortHandle());
        }

        TracingSpan tracingSpan = params.getTracingSpan();
        TracingUtil.registerRequestDataIntoSpan(tracingSpan, post, protocol, IoType.NIO);
        tracingSpan.injectInto(post, TracingUtil::addHeader);

        logger.debug("Executing async request.");
        HttpContext httpCtx = ApacheHttpUtil.createHttpContext(
                params.getExecutionContext(), configMerger);
        httpClient.execute(
                HttpAsyncMethods.create(post),
                new NioMainResponseConsumer(params.getCbFactory(),
                                            params.getExecutionContext(),
                                            httpCtx,
                                            params.getAcceptedTypes(),
                                            params.getAbortHandle(),
                                            tracingSpan,
                                            maxResponseSize),
                httpCtx,
                new FutureCallbackImpl(uri, params.getAbortHandle(), params.getCbFactory()));
    }

    @Override
    public void close() {
        try {
            httpClient.close();
            pool = null;
        } catch (IOException ex) {
            // TODO: use specific exception
            throw new RuntimeException(ex);
        }
    }

    /**
     * Callback which handles errors. For some reason the underlying library
     * does not notify the consumer of connection failures, so we need this
     * callback.
     */
    private static class FutureCallbackImpl implements
            FutureCallback<HttpResponse> {

        private final ResponseCallbackFactory cbFactory;
        private final AbortHandle abortHandle;
        private final String uri;

        public FutureCallbackImpl(String uri,
                                  AbortHandle abortHandle,
                                  ResponseCallbackFactory cbFactory) {
            this.uri = uri;
            this.cbFactory = cbFactory;
            this.abortHandle = abortHandle;
        }

        @Override
        public void completed(HttpResponse result) {
            // nothing to do here - the consumer will handle the result
        }

        @Override
        public void failed(Exception ex) {
            logger.debug("HTTP exchange failed", ex);
            if (cbFactory == null) {
                logger.debug("CbFactory is null - fail will not be propagated");
            }
            cbFactory.failed(ApacheHttpClientExceptionTranslator.translate(
                    ex, abortHandle, uri));
        }

        @Override
        public void cancelled() {
            /*
             * If the transport gets shutdown before we receive the HTTP
             * response (or the HTTP exchange is canceled via the future),
             * report an error to the callback.
             */
            logger.debug("HTTP exchange cancelled");
            if (cbFactory == null) {
                logger.debug("CbFactory is null - fail will not be propagated");
            }
            cbFactory.failed(
                    ApacheHttpClientExceptionTranslator.translate(
                            new CancellationException(), abortHandle));
        }
    }
}
