/* **********************************************************
 * Copyright (c) 2013-2015, 2017-2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.protocol.HttpClientContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.client.exception.InvalidSslCertificateException;
import com.vmware.vapi.client.exception.TransportProtocolException;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.RuntimeData;
import com.vmware.vapi.internal.VersionUtil;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.internal.util.io.IoUtil;
import com.vmware.vapi.protocol.HttpConfiguration;

/**
 * Utility for transport implementations based on the Apache httpcomponents
 * libraries.
 */
public final class ApacheHttpUtil {
    private static final Logger logger =
            LoggerFactory.getLogger(ApacheHttpUtil.class);

    /**
     * Value of the User-Agent HTTP header. Used by client transports.
     */
    public final static String VAPI_USER_AGENT = VersionUtil.getUserAgent();

    private ApacheHttpUtil() {
    }

    /**
     * Validates the HTTP response for a vAPI method invocation.
     *
     * <p>
     * If the validation fails (and exception is reported) this method
     * guarantees that if available the {@code InputStream} returned by
     * {@code httpResponse.getEntity().getContent()} is closed, which means that
     * the TCP connection is returned to the pool. Otherwise, the content of the
     * response is not consumed and the stream is not closed.
     *
     * @param httpResponse HTTP response; must not be <code>null</code>
     * @param acceptedTypes the accepted content types of the request for this
     *        response
     * @throws TransportProtocolException if there is no content in the
     *         response, or the status code is for a redirect, or the
     *         content-type of the response is unexpected
     * @throws InvalidSslCertificateException if the response status is 526
     *         Invalid SSL Certificate
     */
    public static void validateHttpResponse(HttpResponse httpResponse,
                                            Collection<String> acceptedTypes) {
        Validate.notNull(httpResponse);
        HttpEntity entity = httpResponse.getEntity();

        int statusCode = httpResponse.getStatusLine().getStatusCode();
        if (300 <= statusCode && statusCode <= 399) {
            silentClose(entity);
            // redirect is a transport error
            throw new TransportProtocolException(
                    String.format(
                            "HTTP redirect response with status code %d (enable debug logging for details)",
                            statusCode));
        }

        if (entity == null) {
            // no content is a transport error
            throw new TransportProtocolException(
                    String.format(
                            "HTTP response with status code %d and no content (enable debug logging for details)",
                            statusCode));
        }

        String contentType = null;
        if (entity.getContentType() != null) {
            // only consider the media type (ignore "; <parameters>) from
            // the header value
            contentType = stripContentTypeParams(entity.getContentType().getValue());
        }

        if (400 <= statusCode && statusCode <= 599) {
            String text = readResponseBodyAndClose(entity, contentType);

            if (InvalidSslCertificateException.STATUS_CODE == statusCode) {
                throw new InvalidSslCertificateException(text);
            }
            if (text != null) {
                throw new TransportProtocolException(
                        String.format(
                                "HTTP response with status code %d (enable debug logging for details): %s",
                                statusCode, text));
            } else {
                throw new TransportProtocolException(
                        String.format(
                                "HTTP response with status code %d (enable debug logging for details)",
                                statusCode));
            }
        }

        if (contentType == null) {
            silentClose(entity);
            // missing content type of the response
            throw new TransportProtocolException(
                    String.format(
                            "HTTP response with status code %d and no Content-Type header (enable debug logging for details)",
                            statusCode));
        }
        boolean expectedContentType = acceptedTypes.contains(contentType);
        if (!expectedContentType) {
            String text = readResponseBodyAndClose(entity, contentType);
            if (text != null) {
                throw new TransportProtocolException(
                        String.format(
                                "HTTP response with status code %d has unexpected content-type '%s' (enable debug logging for details): %s",
                                statusCode, contentType, text));
            } else {
                throw new TransportProtocolException(
                        String.format(
                                "HTTP response with status code %d has unexpected content-type '%s' (enable debug logging for details)",
                                statusCode, contentType));
            }
        }

        // validation passed - the response body stream is not closed, and can
        // be consumed next
    }

    /**
     * Tries to determine the content type of the given {@code response}.
     * The content type should be contained in the 'Content-Type' HTTP
     * header. If that header is missing, no attempt is made to sniff
     * the content type by examining the content.
     *
     * @param response
     *        the response whose content type must be determined
     *
     * @return if the content type of the response can be determined,
     *         returns the content type; otherwise, returns {@code null}
     */
    public static String getContentType(HttpResponse response) {
        if (response != null) {
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                Header contentType = entity.getContentType();
                if (contentType != null) {
                    return contentType.getValue();
                }
            }
        }
        return null;
    }

    static String stripContentTypeParams(String contentType) {
        if (contentType == null) {
            return null;
        }

        // HTTP 1.1 rfc:
        // Content-Type   = "Content-Type" ":" media-type
        // media-type     = type "/" subtype *( ";" parameter )
        int semiColonIndex = contentType.indexOf(';');
        if (semiColonIndex >= 0) {
            contentType = contentType.substring(0, semiColonIndex);
        }

        // trim it for robustness
        return contentType.trim();
    }

    /**
     * Attempts to interpret an HTTP entity as text.
     *
     * <p>Closes the response body input stream in all cases.
     *
     * @param entity HTTP entity
     * @param contentType content type of the entity
     * @return the text content or <code>null</code>
     */
    private static String readResponseBodyAndClose(HttpEntity entity,
                                                   String contentType) {
        try (InputStream responseBody = entity.getContent()) {
            if (contentType == null) {
                return null;
            }
            if (!contentType.contains("text")) {
                return null;
            }
            return new String(IoUtil.readAll(responseBody), "UTF-8");
        } catch (IOException ex) {
            return "<IOException while reading response body>";
        }
    }

    private static void silentClose(HttpEntity entity) {
        if (entity == null) {
            return;
        }

        try {
            IoUtil.silentClose(entity.getContent());
        } catch (IOException ex) {
            logger.debug("Error while closing HTTP response body stream", ex);
        }
    }

    public static RequestConfig createDefaultRequestConfig(
            HttpConfiguration httpConfig) {
        return RequestConfig.custom()
                .setSocketTimeout(httpConfig.getSoTimeout())
                .setConnectTimeout(httpConfig.getConnectTimeout())
                .build();
    }

    static HttpClientContext createHttpContext(ExecutionContext executionContext,
                          ApacheClientRequestConfigurationMerger configMerger) {
       RuntimeData runtimeData = executionContext == null ? null
               : executionContext.retrieveRuntimeData();
       Integer readTimeout = runtimeData == null ? null : runtimeData.getReadTimeout();
       return createHttpContext(readTimeout, configMerger);
    }

    static HttpClientContext createHttpContext(Integer readTimeout,
                          ApacheClientRequestConfigurationMerger configMerger) {
       RequestConfig config = configMerger.mergeWithDefaultConfig(readTimeout);
       HttpClientContext httpContext = new HttpClientContext();
       httpContext.setRequestConfig(config);
       return httpContext;
    }

}
