/* **********************************************************
 * Copyright (c) 2019-2022 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc.http;

import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.impl.client.LaxRedirectStrategy;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.impl.nio.conn.PoolingNHttpClientConnectionManager;
import org.apache.http.impl.nio.reactor.DefaultConnectingIOReactor;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.apache.http.nio.conn.NoopIOSessionStrategy;
import org.apache.http.nio.conn.SchemeIOSessionStrategy;
import org.apache.http.nio.conn.ssl.SSLIOSessionStrategy;
import org.apache.http.nio.reactor.ConnectingIOReactor;
import org.apache.http.nio.reactor.IOReactorException;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.protocol.client.rpc.http.ConnectionMonitor.CleanableConnectionPool;
import com.vmware.vapi.internal.protocol.client.rpc.http.adapter.NioHttpClientBuilderAdapter;
import com.vmware.vapi.internal.util.DefaultThreadFactory;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.HttpConfiguration;
import com.vmware.vapi.protocol.HttpConfiguration.HeadersProvider;
import com.vmware.vapi.protocol.HttpConfiguration.SslConfiguration;

/**
 * Builder for Apache HTTP NIO Clients.
 */
public class ApacheNioHttpClientBuilder {

    private PoolingNHttpClientConnectionManager connManager;

    private static Logger logger = LoggerFactory.getLogger(ApacheNioHttpClientBuilder.class);
    private CleanableConnectionPool pool;
    private RequestConfig defaultRequestConfig;

    /**
     * @return the default configuration the HTTP client was built with;
     *         {@code null} if it has not been built yet
     */
    public RequestConfig getDefaultRequestConfig() {
       return defaultRequestConfig;
    }

    /**
     * Creates apache HTTP client instance and configures it accordingly to the specified
     * {@code HttpConfiguration}.
     *
     * <p>It is responsibility of the consuming code to properly dispose the client when it is no
     * longer needed.
     *
     * @param httpConfig http settings to be applied
     *
     * @return closable http async client
     *
     * @throws SecurityException
     * If a security manager has been installed and it denies
     * {@link java.net.NetPermission}{@code ("getProxySelector")}
     * when using system default proxy configuration
     * since {@link java.net.ProxySelector#getDefault()} is used in this case.
     */
    public CloseableHttpAsyncClient buildAndConfigure(final HttpConfiguration httpConfig) throws SecurityException {
        Validate.notNull(httpConfig);
        defaultRequestConfig = ApacheHttpUtil.createDefaultRequestConfig(httpConfig);
        try {
            connManager = createConnectionManager(httpConfig);
            HttpAsyncClientBuilder builder = HttpAsyncClients.custom()
               .setConnectionManager(connManager)
               .setUserAgent(ApacheHttpUtil.VAPI_USER_AGENT)
               .setThreadFactory(new DefaultThreadFactory("vAPI-I/O reactor-"))
               .setRedirectStrategy(new LaxRedirectStrategy())
               .setKeepAliveStrategy(new ConnectionKeepAliveStrategy() {
                   @Override
                   public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                       return httpConfig.getKeepAlivePeriod();
                   }
               });
            HeadersProvider headersProvider = httpConfig.getHeadersProvider();
            if (headersProvider != null) {
                builder.addInterceptorFirst(new ApacheClientHeadersProvider(headersProvider));
            }

            HttpConfiguration.ProxyConfiguration proxyConfig = httpConfig.getProxyConfiguration();
            if (proxyConfig != null) {
                HttpClientProxyConfigurationHelper.addProxyConfiguration(
                    proxyConfig,
                    new NioHttpClientBuilderAdapter(builder));
            }

            CloseableHttpAsyncClient client = builder.build();
            client.start();
            return client;
        } catch (IOReactorException ex) {
            // TODO: use specific exception
            throw new RuntimeException(ex);
        }
    }

    /**
     * Registers the connection pool manager only once with a
     * {@code ConnectionMonitor} thread utilized for cleaning stale connections.
     * The returned object needs to be kept as a hard link, in order for
     * successful monitoring to be accomplished.
     *
     * @return CleanableConnectionPool a strong reference pointer
     */
    public CleanableConnectionPool registerClientWithConnectionMonitor() {
        if (connManager == null) {
            throw new IllegalStateException();
        }

        if (pool == null) {
            pool = new CleanableConnectionPool() {
                @Override
                public void closeExpiredConnections() {
                    connManager.closeExpiredConnections();
                }

                @Override
                public int leasedConnections() {
                    return connManager.getTotalStats().getLeased();
                }
            };

            ConnectionMonitor.register(pool);
        } else {
            logger.warn("Unable to register client more than once!");
        }

        return pool;
    }

    private PoolingNHttpClientConnectionManager createConnectionManager(HttpConfiguration config)
            throws IOReactorException {
        ConnectingIOReactor ioreactor = createConnectionIOReactor(config);
        Registry<SchemeIOSessionStrategy> schemeRegistry =
                createAsyncSchemeRegistry(config.getSslConfiguration());
        PoolingNHttpClientConnectionManager cm =
                new PoolingNHttpClientConnectionManager(ioreactor,
                                                        schemeRegistry);
        cm.setMaxTotal(config.getMaxConnections());
        cm.setDefaultMaxPerRoute(config.getMaxConnections());
        return cm;
    }

    private ConnectingIOReactor createConnectionIOReactor(
            HttpConfiguration config) throws IOReactorException {
        IOReactorConfig ioReactorConfig = IOReactorConfig.custom()
                .setIoThreadCount(config.getIoThreadCount())
                .setSoTimeout(config.getSoTimeout())
                .setConnectTimeout(config.getConnectTimeout())
                .build();
        return new DefaultConnectingIOReactor(
                ioReactorConfig, new DefaultThreadFactory("vAPI-I/O dispatcher-"));
    }

    @SuppressWarnings("deprecation")
    Registry<SchemeIOSessionStrategy> createAsyncSchemeRegistry(SslConfiguration sslConfig) {
        RegistryBuilder<SchemeIOSessionStrategy> registryBuilder =
                RegistryBuilder.<SchemeIOSessionStrategy>create()
                               .register(ApacheBioHttpClientBuilder.HTTP_SCHEME,
                                         NoopIOSessionStrategy.INSTANCE);

        if (sslConfig != SslConfiguration.SKIP_SSL_INITIALIZATION) {
            org.apache.http.conn.ssl.X509HostnameVerifier hostnameVerifier =
                    SslClientUtil.createHostnameVerifier(
                                     sslConfig.isHostnameVerificationDisabled(),
                                     SSLIOSessionStrategy.STRICT_HOSTNAME_VERIFIER);
            registryBuilder.register(ApacheBioHttpClientBuilder.HTTPS_SCHEME,
                                     new SSLIOSessionStrategy(SslClientUtil.createSslContext(sslConfig),
                                                              sslConfig.getEnabledProtocols(),
                                                              sslConfig.getEnabledCipherSuites(),
                                                              hostnameVerifier));
        }
        return registryBuilder.build();
    }
}
