/* **********************************************************
 * Copyright (c) 2021 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http;

import java.net.ProxySelector;

import org.apache.http.HttpHost;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.auth.AUTH;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.MalformedChallengeException;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.conn.routing.HttpRoutePlanner;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.SystemDefaultCredentialsProvider;
import org.apache.http.impl.conn.SystemDefaultRoutePlanner;
import org.apache.http.message.BasicHeader;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.protocol.client.rpc.http.adapter.ProxyAwareHttpClientBuilderAdapter;
import com.vmware.vapi.protocol.HttpConfiguration;
import com.vmware.vapi.protocol.UserPassCredentials;
import com.vmware.vapi.protocol.server.rpc.http.InternalException;

class HttpClientProxyConfigurationHelper {
    private static Logger logger = LoggerFactory.getLogger(HttpClientProxyConfigurationHelper.class);

    private static final String PROXY_AUTH_HEADER_VALUE = "Basic";

    /**
     * Adds proxy related configuration need for constructing an http client.
     *
     * @throws SecurityException
     * If a security manager has been installed and it denies
     * {@link java.net.NetPermission}{@code ("getProxySelector")}
     * when calling {@link java.net.ProxySelector#getDefault()}.
     */
    static void addProxyConfiguration(HttpConfiguration.ProxyConfiguration proxyConfig,
                                      ProxyAwareHttpClientBuilderAdapter builderAdapter) throws SecurityException {
        HttpHost defaultProxy = null;
        if (proxyConfig.usesSystemDefaultProxyConfiguration()) {
            logger.debug("Setting up system default proxy configuration...");
            HttpRoutePlanner routePlanner =
                new SystemDefaultRoutePlanner(ProxySelector.getDefault());

            builderAdapter.setRoutePlanner(routePlanner);
        } else {
            // assume default scheme (http)
            defaultProxy = new HttpHost(proxyConfig.getHost(), proxyConfig.getPort());

            logger.debug("Setting default proxy host to {}", defaultProxy.toURI());

            builderAdapter.setProxy(defaultProxy);
        }

        CredentialsProvider credentialsProvider = null;
        UserPassCredentials defaultCredentials =  proxyConfig.getUserPassCredentials();
        if (defaultCredentials != null) {
            credentialsProvider = new BasicCredentialsProvider();
        }
        else if (proxyConfig.usesSystemDefaultProxyConfiguration()) {
            credentialsProvider = new SystemDefaultCredentialsProvider();
        }

        if (credentialsProvider != null) {
            configurePreemptiveBasicAuthentication(
                builderAdapter, defaultProxy, defaultCredentials, credentialsProvider );
        }
    }

    private static void configurePreemptiveBasicAuthentication(
        ProxyAwareHttpClientBuilderAdapter builder,
        HttpHost defaultProxy,
        UserPassCredentials defaultCredentials,
        CredentialsProvider credentialsProvider) {

        BasicScheme proxyAuth = new BasicScheme();
        try {
            proxyAuth.processChallenge(new BasicHeader(AUTH.PROXY_AUTH, PROXY_AUTH_HEADER_VALUE));
        } catch (MalformedChallengeException e) {
            throw new InternalException(e);
        }

        Credentials credentials = (defaultCredentials != null)
            ? new UsernamePasswordCredentials(defaultCredentials.getUser(),
                                              String.valueOf(defaultCredentials.getPassword()))
            : null;

        final BasicAuthCache authCache = new BasicAuthCache();
        HttpRequestInterceptor interceptor;
        if (defaultProxy != null && defaultCredentials != null) {
            logger.debug("Configuring pre-emptive Basic Authentication with scope {} using BasicCredentialsProvider.",
                         defaultProxy.toURI());

            authCache.put(defaultProxy, proxyAuth);
            credentialsProvider.setCredentials(new AuthScope(defaultProxy), credentials);

            interceptor = (request, context) -> {
                ((HttpClientContext) context).setAuthCache(authCache);
                ((HttpClientContext) context).setCredentialsProvider(credentialsProvider);
            };

        } else {
            interceptor = (HttpRequestInterceptor) (request, context) -> {
                HttpHost proxy = (defaultProxy != null)
                    ? defaultProxy
                    : ((HttpClientContext) context).getHttpRoute().getProxyHost();

                if (proxy != null) {
                    logger.debug("Configuring pre-emptive Basic Authentication with scope {} using {}.",
                                 proxy.toURI(),
                                 credentialsProvider.getClass().getSimpleName());

                    authCache.put(proxy, proxyAuth);

                    if (credentialsProvider instanceof BasicCredentialsProvider) {
                        credentialsProvider.setCredentials(new AuthScope(proxy), credentials);
                    }

                    ((HttpClientContext) context).setAuthCache(authCache);
                    ((HttpClientContext) context).setCredentialsProvider(credentialsProvider);
                } else {
                    logger.warn(
                        "Default proxy selector was unable to pick a proxy for target {}! "
                        + "Please check if system properties contain proxyHost/Port for the respective "
                        + "target schema.",
                        ((HttpClientContext) context).getHttpRoute().getTargetHost().toURI());
                }
            };
        }

        builder.addInterceptorFirst(interceptor);
    }
}