/* **********************************************************
 * Copyright 2014, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc.http;

import java.net.Socket;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;

import javax.net.ssl.SSLEngine;
import javax.net.ssl.X509ExtendedKeyManager;
import javax.net.ssl.X509KeyManager;

import com.vmware.vapi.internal.util.Validate;

/**
 * Wraps an instance of X509KeyManager. Chose client certificate based on key
 * alias passed to the constructor.
 */
final class KeyManagerWrapper extends X509ExtendedKeyManager {
    private final X509KeyManager baseKM;
    private final String alias;

    /**
     * Constructs the wrapper with <i>baseKM</i> key manager and for key alias
     * <i>alias</i>
     *
     * @param baseKM
     *           the key manager to use; must not be <code>null</code>
     * @param alias
     *           the alias of the key to be returned
     */
    public KeyManagerWrapper(X509KeyManager baseKM, String alias) {
        Validate.notNull(baseKM);
        this.baseKM = baseKM;
        this.alias = alias;
    }

    /**
     * Look at the valid certificates in the key store for one with matching
     * alias. If null was passed as alias in the constructor call to the wrapped
     * key manager will be performed.
     *
     * @see javax.net.ssl.X509KeyManager#chooseClientAlias(String[], Principal[],
     *      Socket)
     */
    @Override
    public String chooseClientAlias(String[] keyType, Principal[] issuers,
          Socket socket) {
       if (null == alias) {
          return baseKM.chooseClientAlias(keyType, issuers, socket);
       }

       if (isAliasFound(keyType, issuers)) {
          return alias;
       } else {
          return null;
       }
    }

    @Override
    public String chooseServerAlias(String keyType, Principal[] issuers,
                                    Socket socket) {
        return baseKM.chooseServerAlias(keyType, issuers, socket);
    }

    @Override
    public X509Certificate[] getCertificateChain(String alias) {
        return baseKM.getCertificateChain(alias);
    }

    @Override
    public String[] getClientAliases(String keyType, Principal[] issuers) {
        return baseKM.getClientAliases(keyType, issuers);
    }

    @Override
    public PrivateKey getPrivateKey(String alias) {
        return baseKM.getPrivateKey(alias);
    }

    @Override
    public String[] getServerAliases(String keyType, Principal[] issuers) {
        return baseKM.getServerAliases(keyType, issuers);
    }

    /*
     * Introduced this due to some SSL handshake difference between VC and java
     * servers. There was a failure when using NIO client trying to communicate
     * with VC. The client keystore was not used at all. With this fix - the
     * client keystore is used correctly and the VC doesn't fail to validate get
     * the client certificate. We has a unit test for NIO/BIO clients with
     * keystore against server with truststore, but this tests doesn't catch the
     * failure. The issue was observed when calling the VC
     * SessionManager.loginExtensionByCertificate(...).
     */
    @Override
    public String chooseEngineClientAlias(String[] keyType,
                                          Principal[] issuers,
                                          SSLEngine sslEngine) {
        if (null == alias || !isAliasFound(keyType, issuers)) {
            return super.chooseEngineClientAlias(keyType, issuers, sslEngine);
        } else {
            return alias;
        }
    }

    private boolean isAliasFound(String[] keyType, Principal[] issuers) {
        for (int i = 0; i < keyType.length; ++i) {
            String[] validAliases =
                    baseKM.getClientAliases(keyType[i], issuers);
            if (validAliases != null) {
                for (int j = 0; j < validAliases.length; ++j) {
                    if (validAliases[j].equals(alias)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }
}
