/* **********************************************************
 * Copyright (c) 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc.http.handle;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.core.abort.AbortHandle;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.ResponseCallback;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.TransportControl;
import com.vmware.vapi.internal.protocol.common.Util;
import com.vmware.vapi.internal.protocol.common.http.ApacheHttpClientExceptionTranslator;
import com.vmware.vapi.internal.protocol.common.http.BinaryInput;
import com.vmware.vapi.internal.protocol.common.http.FrameDeserializer;
import com.vmware.vapi.internal.protocol.common.http.impl.ChunkedTransferEncodingFrameDeserializer;
import com.vmware.vapi.internal.protocol.common.http.impl.InputStreamBinaryInput;
import com.vmware.vapi.internal.util.io.IoUtil;

public class BioStreamingResponseHandlingStrategy
        implements BioResponseHandlingStrategy {

    private static Logger logger = LoggerFactory
            .getLogger(BioStreamingResponseHandlingStrategy.class);

    @Override
    public void handleContent(InputStream content,
                              ResponseCallback cb,
                              AbortHandle abortHandle,
                              String uri)
            throws IOException {
        ControllableTransport transport = new ControllableTransport(content,
                                                                    cb,
                                                                    abortHandle,
                                                                    uri);
        Util.registerStreamingAbortListerner(transport, abortHandle);
        if (Util.checkRequestAborted(abortHandle, cb)) {
            transport.cancel();
            return;
        }
        transport.resumeRead();
    }

    public static class ControllableTransport implements TransportControl {
        // multiple response frames
        private final FrameDeserializer frameDeserializer = new ChunkedTransferEncodingFrameDeserializer();
        private InputStream is;
        private ResponseCallback cb;
        private AbortHandle abortHandle;
        private String uri;

        private BinaryInput input;
        private volatile boolean readSuspended = false;

        // Indicates whether the streaming is aborted
        private volatile boolean aborted = false;

        // Used for ensuring that streaming is not closed
        // while reading, when an abort is issued
        private ReentrantLock lock;

        public ControllableTransport(InputStream is,
                                     ResponseCallback cb,
                                     AbortHandle abortHandle,
                                     String uri) {
            this.is = is;
            this.cb = cb;
            this.abortHandle = abortHandle;
            this.uri = uri;

            this.input = new InputStreamBinaryInput(is);
            this.lock = new ReentrantLock();
        }

        @Override
        protected void finalize() throws Throwable {
            IoUtil.silentClose(is);
            super.finalize();
        }

        @Override
        public void suspendRead() {
            readSuspended = true;
        }

        @Override
        public void resumeRead() {
            // Unwind stack - will resume on onSuccess
            if (lock.isHeldByCurrentThread()) {
                readSuspended = false;
                logger.trace("The same thread is already reading frames.");
                return;
            }

            lock.lock();
            readSuspended = false;
            byte[] frame = null;
            try {
                while (!readSuspended && !aborted) {
                    logger.trace("Reading frames.");
                    frame = frameDeserializer.readFrame(input);
                    if (frame == null || frame.length == 0) {
                        logger.trace("Finished reading.");
                        break;
                    }
                    cb.received(new ByteArrayInputStream(frame), this);
                }
                if (((frame == null || frame.length == 0) && !readSuspended)
                    || aborted) {
                    IoUtil.silentClose(is);
                    is = null;
                    cb.completed();
                }
            } catch (Exception ex) {
                logger.debug("Error during reading frames.", ex);
                cb.failed(ApacheHttpClientExceptionTranslator
                        .translate(ex, abortHandle, uri));
                IoUtil.silentClose(is);
                is = null;
            } finally {
                lock.unlock();
            }
        }

        @Override
        public void cancel() {
            logger.debug("Cancelling the streaming frame desirialization.");
            aborted = true;
            if (lock.tryLock()) {
                IoUtil.silentClose(is);
                is = null;
                lock.unlock();
            }
        }
    }
}
