/* **********************************************************
 * Copyright (c) 2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http.handle;

import java.io.IOException;
import java.nio.BufferOverflowException;

import org.apache.http.ContentTooLongException;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.ContentDecoder;
import org.apache.http.nio.IOControl;
import org.apache.http.nio.entity.ContentBufferEntity;
import org.apache.http.nio.protocol.AbstractAsyncResponseConsumer;
import org.apache.http.nio.protocol.BasicAsyncResponseConsumer;
import org.apache.http.nio.util.HeapByteBufferAllocator;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.Asserts;

/**
 * A modified copy of apache {@link BasicAsyncResponseConsumer} that adds the
 * ability to set a limit of the size of the response that it can consume by
 * replacing the underlining buffer
 */
public class CappedAsyncResponseConsumer
        extends AbstractAsyncResponseConsumer<HttpResponse> {

    private static final int DEFAULT_INIT_BUFF_SIZE = 2 << 11; // 4 kib
    // Taken from BasicAsyncResponseConsumer. We have a maximum for the initial
    // buffer size for quicker allocation.
    private static final int MAX_INIT_BUFF_SIZE = 2 << 17; // 256 kib
    private volatile HttpResponse response;
    protected volatile CappedInputBuffer buf;
    private final int maxTotalBuffSize;

    public CappedAsyncResponseConsumer(int maxBuffSize) {
        super();
        if (maxBuffSize <= 0) {
            throw new IllegalArgumentException("maxBuffSize must be a positive number");
        }
        this.maxTotalBuffSize = maxBuffSize;
    }

    @Override
    protected void onResponseReceived(final HttpResponse response)
            throws IOException {
        this.response = response;
    }

    @Override
    protected void onEntityEnclosed(final HttpEntity entity, final ContentType contentType)
            throws IOException {
        long len = entity.getContentLength();
        if (len > maxTotalBuffSize) {
            throw new ContentTooLongException("Entity content is too long: %d. Max allowed is: %d",
                                              len,
                                              maxTotalBuffSize);
        }
        int initBuffSize = 0;
        if (len > 0) {
            // content length is known so allocate exact buff size at once if it
            // is less than the max initial buff size. In this case memory
            // re-allocation is not expected.
            initBuffSize = Math.min(MAX_INIT_BUFF_SIZE, (int) len);
        } else {
            initBuffSize = Math.min(DEFAULT_INIT_BUFF_SIZE, maxTotalBuffSize);
        }
        this.buf = new CappedInputBuffer(maxTotalBuffSize,
                                         initBuffSize,
                                         new HeapByteBufferAllocator());
        this.response.setEntity(new ContentBufferEntity(entity, this.buf));
    }

    @Override
    protected void onContentReceived(final ContentDecoder decoder,
                                     final IOControl ioControl)
            throws IOException {
        Asserts.notNull(this.buf, "Content buffer");
        try {
            this.buf.consumeContent(decoder);
        } catch (BufferOverflowException ex) {
            throw new IOException("Cannot consume response. Buffer overflow");
        }
    }

    @Override
    protected void releaseResources() {
        this.response = null;
        this.buf = null;
    }

    @Override
    protected HttpResponse buildResult(final HttpContext context) {
        return this.response;
    }
}