/* **********************************************************
 * Copyright (c) 2016, 2021 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.common;

import static com.vmware.vapi.internal.protocol.common.XsdDatatypeConverter.parseBoolean;

import java.util.LinkedList;

import com.vmware.vapi.data.BooleanValue;
import com.vmware.vapi.data.DataType;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.DoubleValue;
import com.vmware.vapi.data.IntegerValue;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.protocol.common.json.JsonInvalidDataValueException;

/**
 * Defines class capable of creating data values.
 */
public class DataValueBuilder {
    // TODO: what to use for stuct name in DV'?
    public static final String STRUCT_NAME = "struct";

    private static interface PropertyAppender {
        void addDataValue(String name, DataValue dv);
    }

    private final LinkedList<PropertyAppender> appenders = new LinkedList<>();
    private DataValue result;

    /**
     * Starts the writing of the structure.
     */
    public void start() {
        PropertyAppender appender = new PropertyAppender() {
            @Override
            public void addDataValue(String name, DataValue dv) {
                result = dv;
            }
        };
        appenders.add(appender);
    }

    /**
     * Ends processing.
     */
    public void end() {
        pollContext();
    }

    /**
     * Records property in the current context. Property can be element in array
     * or field in structure.
     */
    public void addProperty(String name, String value, DataType dataType) {
        appendDataValue(name, buildPrimitiveValue(value, dataType));
    }

    /**
     * Begins recoding an array property.
     *
     * @param propertyName the name of the array property.
     */
    public void startArray(String propertyName) {
        final ListValue listValue = new ListValue();
        appendDataValue(propertyName, listValue);
        appenders.add(new PropertyAppender() {
            @Override
            public void addDataValue(String name, DataValue dv) {
                listValue.add(dv);
            }
        });
    }

    /**
     * Ends recoding of the array.
     */
    public void endArray() {
        pollContext();
    }

    /**
     * Starts struct property.
     *
     * @param propertyName the name of the property.
     */
    public void startStruct(String propertyName) {
        final StructValue structValue = new StructValue(STRUCT_NAME);
        appendDataValue(propertyName, structValue);
        appenders.add(new PropertyAppender() {
            @Override
            public void addDataValue(String name, DataValue dv) {
                structValue.setField(name, dv);
            }
        });
    }

    /**
     * Ends the current structure.
     */
    public void endStruct() {
        pollContext();
    }

    public DataValue getResult() {
        return result;
    }

    private void appendDataValue(String name, DataValue dataValue) {
        this.appenders.getLast().addDataValue(name, dataValue);
    }

    private DataValue buildPrimitiveValue(String value, DataType targetType) {
        switch (targetType) {
            case STRING:
                return new StringValue(value);
            case INTEGER:
                return new IntegerValue(Long.parseLong(value));
            case DOUBLE:
                return new DoubleValue(Double.parseDouble(value));
            case BOOLEAN:
                return parseBoolean(value) ? BooleanValue.TRUE : BooleanValue.FALSE;
            case OPTIONAL:
                return OptionalValue.UNSET;
            default:
                throw new JsonInvalidDataValueException("Unsupported data type: " + targetType);
        }
    }

    private void pollContext() {
        appenders.pollLast();
    }
}
