/* **********************************************************
 * Copyright (c) 2013-2014, 2018, 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.common.http;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.concurrent.CancellationException;

import org.apache.http.ConnectionClosedException;
import org.apache.http.HttpException;
import org.apache.http.ParseException;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.client.exception.ConnectionException;
import com.vmware.vapi.client.exception.InternalClientException;
import com.vmware.vapi.client.exception.SslException;
import com.vmware.vapi.client.exception.TransportProtocolException;
import com.vmware.vapi.internal.common.exception.ExceptionTranslatorBase;
import com.vmware.vapi.internal.core.abort.AbortHandle;
import com.vmware.vapi.internal.core.abort.RequestAbortedException;

public class ApacheHttpClientExceptionTranslator extends ExceptionTranslatorBase {
    /**
     * Translate 3rd party exception to our own exception hierarchy of transport
     * exceptions. The exception stacktrace is fixed as if it was created in the
     * calling method.
     *
     * @param e the exception to be translated; must not be null
     * @param abortHandle abort handle for the request translating the exception
     * @return the translated exception
     */
    public static RuntimeException translate(Exception e,
                                             AbortHandle abortHandle) {
      return translate(e, abortHandle, null);
    }

    /**
     * Translate 3rd party exception to our own exception hierarchy of transport
     * exceptions. The exception stacktrace is fixed as if it was created in the
     * calling method.
     *
     * @param e the exception to be translated; must not be null
     * @param abortHandle abort handle for the request translating the exception
     * @param uri the uri to add to the exception message; could be {@code null}.
     *        Applies only for IO and SSL exceptions
     * @return the translated exception
     */
    public static RuntimeException translate(Exception e,
                                             AbortHandle abortHandle,
                                             String uri) {
        if (abortHandle != null && abortHandle.isAborted()) {
            return new RequestAbortedException(e.getMessage(), e);
        }

        String message = getUriExceptionMessage(e, uri);
        if (e instanceof javax.net.ssl.SSLException ||
            isBouncyCastleTlsException(e)) {
            return fixStackTrace(new SslException(message, e));
        }

        if (e instanceof SocketException ||
            e instanceof SocketTimeoutException ||
            e instanceof ConnectionClosedException ||
            e instanceof InterruptedIOException ||
            e instanceof UnknownHostException) {
            return fixStackTrace(new ConnectionException(message, e));
        }

        if (e instanceof org.apache.http.MalformedChunkCodingException ||
            e instanceof HttpException ||
            e instanceof ParseException ||
            e instanceof CancellationException ||
            e instanceof CoreException ||
            e instanceof IOException) {
            return fixStackTrace(new TransportProtocolException(message, e));
        }

        if (e instanceof RuntimeException) {
            return (RuntimeException) e;
        }

        return new InternalClientException(e.getMessage(), e);
    }

    public static String getUriExceptionMessage(Exception e, String uri) {
        return uri == null ? e.getMessage()
                           : String.format("%s invocation failed with \"%s\"",
                                           uri,
                                           e);
    }

    static boolean isBouncyCastleTlsException(Exception e) {
        // bctls-fips 1.0.10 reports its own instaces of IOException which are
        // not extending from javax.net.ssl.SSLException, could have done better
        // but let us handle (w/out loading any BC code):
        //   - org.bouncycastle.tls.TlsException
        //   - org.bouncycastle.tls.TlsFatalAlert
        //   - org.bouncycastle.tls.TlsFatalAlertReceived
        //   - org.bouncycastle.tls.crypto.TlsCryptoException
        return e instanceof IOException &&
                e.getClass().getName().startsWith("org.bouncycastle.tls") &&
                e.getClass().getSimpleName().startsWith("Tls");
    }
}
