/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.common.http.impl;

import java.nio.ByteBuffer;

import com.vmware.vapi.internal.protocol.common.http.BinaryInput;
import com.vmware.vapi.internal.util.Validate;

/**
 * Handle to an input source, which wraps a {@link java.nio.ByteBuffer}.
 * Logically this handle is just a piece of the input source. The input source
 * never gets depleted, the handle would never return
 * {@link BinaryInput#NO_DATA_PERMANENT}.
 * <p>
 * <i>Thread-safety:</i> not thread-safe.
 * </p>
 */
public final class ByteBufferBinaryInput implements BinaryInput {

    private final ByteBuffer buffer;

    public ByteBufferBinaryInput(ByteBuffer buffer) {
        Validate.notNull(buffer);
        this.buffer = buffer;
    }

    @Override
    public int read() {
        if (buffer.remaining() > 0) {
            return buffer.get();
        } else {
            return BinaryInput.NO_DATA_TEMPORARY;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) {
        int remaining = buffer.remaining();
        if (remaining > 0) {
            int effectiveLen = Math.min(len, remaining);
            buffer.get(b, off, effectiveLen);
            return effectiveLen;
        } else {
            return BinaryInput.NO_DATA_TEMPORARY;
        }
    }

}
