/* **********************************************************
 * Copyright 2012-2013, 2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.common.http.impl;

import java.io.IOException;

import com.vmware.vapi.internal.protocol.common.http.BinaryInput;
import com.vmware.vapi.internal.protocol.common.http.FrameDeserializer;

/**
 * <p>
 * Frame deserializer which uses a format similar to HTTP chunked transfer
 * encoding.
 * </p>
 * <p>
 * The encoding of a single frame is [frame-length][CRLF][frame-payload][CRLF].
 * The frame length is a non-empty sequence of hexadecimal digits which
 * represents the number of bytes in the frame payload. The frame payload is
 * binary data. CRLF is a carriage return (ASCII 13) followed by a line feed
 * (ASCII 10).
 * </p>
 * <p>
 * <i>Thread-safety:</i> not thread-safe.
 * </p>
 */
public final class ChunkedTransferEncodingFrameDeserializer implements
        FrameDeserializer {

    /**
     * Don't deserialize frames larger than 4GB.
     * length of 4GB ~ 32 binary digits ~ 8 hex digits
     */
    private static final int MAX_HEX_DIGITS_IN_LEN_FIELD = 8;

    /**
     * Finite state machine for the frame.
     */
    private enum State {

        /**
         * Initial state. First field is length.
         */
        LENGTH,

        /**
         * After length comes the data field.
         */
        DATA,

        /**
         * After the data field must be a '\r'.
         */
        CR,

        /**
         * After '\r' must come a '\n'.
         */
        LF,

        /**
         * Final state. A whole frame is ready.
         */
        FRAME
    }

    private State state = State.LENGTH;

    private LengthFieldDeserializer lengthDeserializer;

    private int length;

    private FixedLengthFieldDeserializer fixedLengthFieldDeserializer;

    private byte[] payload;

    private int maxFrameSize = 0;

    public ChunkedTransferEncodingFrameDeserializer() {
    }

    /**
     * @param maxFrameSize The maximum allowed size of the frame that is
     *        received. 0 means that it is unlimited
     */
    public ChunkedTransferEncodingFrameDeserializer(int maxFrameSize) {
        if (maxFrameSize < 0) {
            throw new IllegalArgumentException("maxFrameSize cannot be negative");
        }
        this.maxFrameSize = maxFrameSize;
    }

    @Override
    public byte[] readFrame(BinaryInput inp) throws IOException {
        byte[] result = null;
        boolean enoughData = true;
        while (enoughData && result == null) {
            switch (state) {
            case LENGTH:
                enoughData = atLength(inp);
                break;
            case DATA:
                enoughData = atData(inp);
                break;
            case CR:
                enoughData = atCr(inp);
                break;
            case LF:
                enoughData = atLf(inp);
                break;
            case FRAME:
                result = payload;
                break;
            }
        }
        if (result != null) {
            state = State.LENGTH;
            length = -1;
            payload = null;
        }
        return result;
    }

    private boolean atLength(BinaryInput inp) throws IOException {
        if (lengthDeserializer == null) {
            lengthDeserializer = new LengthFieldDeserializer(
                    MAX_HEX_DIGITS_IN_LEN_FIELD);
        }
        Integer len = lengthDeserializer.read(inp);
        if (len == null) {
            return false;
        }
        if (maxFrameSize > 0 && len > maxFrameSize) {
            throw new IOException(String
                    .format("Frame size %d is bigger than allowed size %d",
                            len,
                            maxFrameSize));
        }
        lengthDeserializer = null;
        length = len;
        state = State.DATA;
        return true;
    }

    private boolean atData(BinaryInput inp) throws IOException {
        if (fixedLengthFieldDeserializer == null) {
            fixedLengthFieldDeserializer = new FixedLengthFieldDeserializer(length);
        }
        byte[] frame = fixedLengthFieldDeserializer.read(inp);
        if (frame == null) {
            return false;
        }
        fixedLengthFieldDeserializer = null;
        payload = frame;
        state = State.CR;
        return true;
    }

    private boolean atCr(BinaryInput inp) throws IOException {
        int b = inp.read();
        if (b == BinaryInput.NO_DATA_TEMPORARY) {
            return false;
        }
        if (b == BinaryInput.NO_DATA_PERMANENT) {
            throw new IOException(
                    "Unexpected end of stream while reading frame trailer.");
        }
        if (b != '\r') {
            throw new IOException(
                    "Expected carriage-return ASCII character. Found " + b);
        }
        state = State.LF;
        return true;
    }

    private boolean atLf(BinaryInput inp) throws IOException {
        int b = inp.read();
        if (b == BinaryInput.NO_DATA_TEMPORARY) {
            return false;
        }
        if (b == BinaryInput.NO_DATA_PERMANENT) {
            throw new IOException(
                    "Unexpected end of stream while reading frame trailer.");
        }
        if (b != '\n') {
            throw new IOException("Expected line-feed ASCII character. Found "
                    + b);
        }
        state = State.FRAME;
        return true;
    }

}
