/* **********************************************************
 * Copyright (c) 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.common.json;

import java.util.HashMap;
import java.util.Map;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.ApplicationData;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.internal.security.SecurityUtil;

/**
 * Provides a way to keep the immutability aspect of the
 * {@link ExecutionContext}, while keeping the copying of hash-maps during
 * JSON request-building to a minimum.
 */
public class JsonContextBuilderRequestParams extends JsonInvokeRequestParams2 {
    private final ExecutionContextBuilder ctxBuilder;

    JsonContextBuilderRequestParams(String serviceId,
                             String operationId,
                             ExecutionContextBuilder ctxBuilder,
                             DataValue input) {
        super(serviceId, operationId, null, input);
        this.ctxBuilder = ctxBuilder;
    }

    @Override
    public ExecutionContext getCtx() {
        return ctxBuilder.view();
    }

    public ExecutionContextBuilder getCtxBuilder() {
        return ctxBuilder;
    }

    /**
     * Allows building an execution context gradually without requiring
     * incessant copying of the hash-maps involved.
     */
    public static class ExecutionContextBuilder {
        public final Map<String, Object> security = new HashMap<>();
        public final Map<String, String> applicationData = new HashMap<>();
        private final LazyExecutionContext lazyCtx = new LazyExecutionContext();

        /**
         * @return an of {@link ExecutionContext} with pre-instated
         *         application-data and security-context, suitable for passing
         *         to a layer outside of JSON request building
         */
        public ExecutionContext build() {
            ApplicationData appData = new ApplicationData(applicationData);
            SecurityContext secCtx = security.isEmpty() ? null
                    : SecurityUtil.createDefaultSecurityContext(security);
            return new ExecutionContext(appData, secCtx);
        }

        /**
         * @return an instance of ExecutionContext that would allocate memory
         *         only upon retrievals of the underlying application-data and
         *         security-context
         */
        public ExecutionContext view() {
            return lazyCtx;
        }

        /**
         * ExecutionContext that would allocate memory only upon retrievals of
         * the underlying application-data and security-context.
         */
        private class LazyExecutionContext extends ExecutionContext {
            private static final long serialVersionUID = 1L;

            @Override
            public ApplicationData retrieveApplicationData() {
                return new ApplicationData(applicationData);
            }

            @Override
            public SecurityContext retrieveSecurityContext() {
                return security.isEmpty() ? null
                        : SecurityUtil.createDefaultSecurityContext(security);
            }
        }
    }
}

