/* **********************************************************
 * Copyright (c) 2017 VMware, Inc. All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.provider;

import com.vmware.vapi.bindings.server.InvocationContext;
import com.vmware.vapi.bindings.server.impl.InvocationContextImpl;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.MethodDefinition;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataDefinition;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StringDefinition;
import com.vmware.vapi.internal.util.TaskUtil;
import com.vmware.vapi.provider.ApiMethod;

/**
 * The <code>ApiTaskMethod</code> class provides introspection
 * and invocation APIs for the task method of an operation annotated with
 * "@Task" or "@TaskOnly" in VMODL2 definition. An operation annotated with
 * "@Task" has both non-task method and task method while an operation
 * annotated with "@TaskOnly" has only task method. The name of a task method
 * is the operation name plus "$task" suffix.
 */
public class ApiTaskMethod implements ApiMethod {
    final private ApiMethod method;
    final private MethodDefinition definition;

    public ApiTaskMethod(ApiMethod method) {
        this.method = method;
        MethodIdentifier id = new MethodIdentifier(
                method.getIdentifier().getInterfaceIdentifier(),
                method.getIdentifier().getName() + TaskUtil.TASK);

        definition = new MethodDefinition(id,
                method.getInputDefinition(),
                StringDefinition.getInstance(),
                method.getDefinition().getErrorDefinitions(),
                method.getDefinition().getTaskSupport());

    }

    @Override
    public MethodIdentifier getIdentifier() {
        return definition.getIdentifier();
    }

    @Override
    public MethodDefinition getDefinition() {
        return definition;
    }

    @Override
    public DataDefinition getInputDefinition() {
        return definition.getInputDefinition();
    }

    @Override
    public DataDefinition getOutputDefinition() {
        return definition.getOutputDefinition();
    }

    @Override
    public void invoke(InvocationContext ctx,
                       DataValue input,
                       AsyncHandle<MethodResult> asyncHandle) {
        InvocationContext invocationContext =
                new InvocationContextImpl(ctx.getExecutionContext(),
                        true);
        method.invoke(invocationContext, input, asyncHandle);
    }
}