/* **********************************************************
 * Copyright 2013-2014, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.provider.introspection;

import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.OPERATION_GET_METHOD_ID;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.OPERATION_GET_METHOD_OPERATION_ID_PARAM;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.OPERATION_GET_METHOD_SERVICE_ID_PARAM;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.PROVIDER_GET_METHOD_ID;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.SERVICE_GET_METHOD_ID;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.SERVICE_GET_METHOD_ID_PARAM;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.SERVICE_LIST_METHOD_ID;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.createMethodInput;

import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.client.exception.MessageProtocolException;
import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.AsyncHandleSyncAdapter;
import com.vmware.vapi.core.ErrorValueException;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.InterfaceDefinition;
import com.vmware.vapi.core.InterfaceIdentifier;
import com.vmware.vapi.core.MethodDefinition;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.core.ProviderDefinition;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.provider.introspection.ApiIntrospectionStub;
import com.vmware.vapi.std.StandardDataFactory;

/**
 * Implementation of {@link ApiIntrospectionStub} which does not rely on
 * code-generated bindings.
 */
public final class ApiIntrospectionStubImpl implements ApiIntrospectionStub {

    private final ApiProvider provider;

    private static Logger logger =
            LoggerFactory.getLogger(ApiIntrospectionStubImpl.class);

    public ApiIntrospectionStubImpl(ApiProvider provider) {
        this.provider = provider;
    }

    private static ExecutionContext newCtx() {
        return new ExecutionContext();
    }

    @Override
    public ProviderDefinition getDefinition() {
        return getDefinition(newCtx());
    }

    @Override
    public Set<InterfaceIdentifier> getInterfaceIdentifiers() {
        return getInterfaceIdentifiers(newCtx());
    }

    @Override
    public InterfaceDefinition getInterface(InterfaceIdentifier ifaceId) {
        return getInterface(newCtx(), ifaceId);
    }

    @Override
    public MethodDefinition getMethod(MethodIdentifier methodId) {
        return getMethod(newCtx(), methodId);
    }

    @Override
    public ProviderDefinition getDefinition(ExecutionContext ctx) {
        AsyncHandleSyncAdapter<ProviderDefinition> handle =
                new AsyncHandleSyncAdapter<ProviderDefinition>();
        getDefinition(ctx, handle);
        return handle.get();
    }

    @Override
    public Set<InterfaceIdentifier> getInterfaceIdentifiers(
            ExecutionContext ctx) {
        AsyncHandleSyncAdapter<Set<InterfaceIdentifier>> handle =
                new AsyncHandleSyncAdapter<Set<InterfaceIdentifier>>();
        getInterfaceIdentifiers(ctx, handle);
        return handle.get();
    }

    @Override
    public InterfaceDefinition getInterface(ExecutionContext ctx,
            InterfaceIdentifier ifaceId) {
        AsyncHandleSyncAdapter<InterfaceDefinition> handle =
                new AsyncHandleSyncAdapter<InterfaceDefinition>();
        getInterface(ctx, ifaceId, handle);
        return handle.get();
    }

    @Override
    public MethodDefinition getMethod(ExecutionContext ctx,
            MethodIdentifier methodId) {
        AsyncHandleSyncAdapter<MethodDefinition> handle =
                new AsyncHandleSyncAdapter<MethodDefinition>();
        getMethod(ctx, methodId, handle);
        return handle.get();
    }

    @Override
    public void getDefinition(ExecutionContext ctx,
            AsyncHandle<ProviderDefinition> asyncHandle) {
        Validate.notNull(ctx);
        Validate.notNull(asyncHandle);
        MethodIdentifier method = PROVIDER_GET_METHOD_ID;
        StructValue input = createMethodInput();
        AsyncHandle<MethodResult> cb;
        cb = new AsyncHandleImpl<ProviderDefinition>(asyncHandle) {
            @Override
            protected ProviderDefinition convert(DataValue result) {
                return IntrospectionDataFactory.fromProviderInfo(result);
            }
        };
        provider.invoke(method.getInterfaceIdentifier().getName(),
                        method.getName(),
                        input,
                        ctx,
                        cb);
    }

    @Override
    public void getInterfaceIdentifiers(ExecutionContext ctx,
            AsyncHandle<Set<InterfaceIdentifier>> asyncHandle) {
        Validate.notNull(ctx);
        Validate.notNull(asyncHandle);
        MethodIdentifier method = SERVICE_LIST_METHOD_ID;
        StructValue input = createMethodInput();
        AsyncHandle<MethodResult> cb;
        cb = new AsyncHandleImpl<Set<InterfaceIdentifier>>(asyncHandle) {
            @Override
            protected Set<InterfaceIdentifier> convert(DataValue result) {
                return IntrospectionDataFactory.fromServiceIdList(result);
            }
        };
        provider.invoke(method.getInterfaceIdentifier().getName(),
                        method.getName(),
                        input,
                        ctx,
                        cb);
    }

    @Override
    public void getInterface(ExecutionContext ctx,
            final InterfaceIdentifier ifaceId,
            AsyncHandle<InterfaceDefinition> asyncHandle) {
        Validate.notNull(ctx);
        Validate.notNull(ifaceId);
        Validate.notNull(asyncHandle);
        MethodIdentifier method = SERVICE_GET_METHOD_ID;
        StructValue input = createMethodInput();
        input.setField(SERVICE_GET_METHOD_ID_PARAM, ifaceId.getName());
        AsyncHandle<MethodResult> cb;
        cb = new AsyncHandleImpl<InterfaceDefinition>(asyncHandle) {
            @Override
            protected InterfaceDefinition convert(DataValue result) {
                return IntrospectionDataFactory
                        .fromServiceInfo(result, ifaceId);
            }

            @Override
            public void setResult(MethodResult result) {
                // NotFound error has to be translated to null result
                if (!result.success()
                        && StandardDataFactory.NOT_FOUND.equals(result
                                .getError().getName())) {
                    handle.setResult(null);
                } else {
                    super.setResult(result);
                }
            }
        };
        provider.invoke(method.getInterfaceIdentifier().getName(),
                        method.getName(),
                        input,
                        ctx,
                        cb);
    }

    @Override
    public void getMethod(ExecutionContext ctx,
            final MethodIdentifier methodId,
            AsyncHandle<MethodDefinition> asyncHandle) {
        Validate.notNull(ctx);
        Validate.notNull(methodId);
        Validate.notNull(asyncHandle);
        MethodIdentifier method = OPERATION_GET_METHOD_ID;
        StructValue input = createMethodInput();
        input.setField(OPERATION_GET_METHOD_SERVICE_ID_PARAM,
                methodId.getInterfaceIdentifier().getName());
        input.setField(OPERATION_GET_METHOD_OPERATION_ID_PARAM,
                methodId.getName());
        AsyncHandle<MethodResult> cb;
        cb = new AsyncHandleImpl<MethodDefinition>(asyncHandle) {
            @Override
            protected MethodDefinition convert(DataValue result) {
                return IntrospectionDataFactory
                        .fromOperationInfo(result, methodId);
            }

            @Override
            public void setResult(MethodResult result) {
                // NotFound error has to be translated to null result
                if (!result.success()
                        && StandardDataFactory.NOT_FOUND.equals(result
                                .getError().getName())) {
                    handle.setResult(null);
                } else {
                    super.setResult(result);
                }
            }
        };
        provider.invoke(method.getInterfaceIdentifier().getName(),
                        method.getName(),
                        input,
                        ctx,
                        cb);
    }

    /**
     * Adapts a type-specific result handle as a generic result handle.
     */
    private static abstract class AsyncHandleImpl<T> extends
            AsyncHandle<MethodResult> {

        protected final AsyncHandle<T> handle;

        public AsyncHandleImpl(AsyncHandle<T> handle) {
            this.handle = handle;
        }

        /**
         * Converts the result of the handle from value to binding.
         *
         * @param result result of the handle as value
         * @return result of the handle as binding object
         * @throws IntrospectionException format error coming from
         *          {@link IntrospectionDataFactory}
         */
        protected abstract T convert(DataValue result);

        @Override
        public void updateProgress(DataValue progress) {
            handle.updateProgress(progress);
        }

        @Override
        public void setResult(MethodResult result) {
            if (result.success()) {
                try {
                    tryReadHttpTerminalFrame(result);
                    handle.setResult(convert(result.getOutput()));
                } catch (RuntimeException ex) {
                    // translate to client exception
                    handle.setError(new MessageProtocolException(ex));
                }
            } else {
                handle.setError(new ErrorValueException(result.getError()));
            }
        }

        @Override
        public void setError(RuntimeException error) {
            handle.setError(error);
        }

        /*
         * We are sure that this is a "Mono" response, since there is no
         * Publisher. Therefore, we are ought to initiate a read, which will go
         * through the HTTP terminal frame, hence, closing the underlying TCP
         * connection.
         */
        private void tryReadHttpTerminalFrame(MethodResult methodResult) {
            try {
                logger.debug("Attempting to close the TCP connection.");
                if (methodResult != null && methodResult.getNext() != null) {
                    logger.debug("Reading HTTP terminal frame.");
                    // <null> is supplied to forbid any further readings.
                    methodResult.getNext().accept(null);
                }
            } catch (Exception e) {
                logger.warn("Failed to read HTTP terminal frame. Connection not closed.",
                            e);
            }
        }
    }
}
