/* **********************************************************
 * Copyright 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.tracing;

import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.internal.tracing.otel.TracingAttributeKey;
import com.vmware.vapi.tracing.TracingLevel;

/**
 * An interface that represents a tracing span.
 */
public interface TracingSpan {

    TracingSpan NO_OP = new NoopSpan();

    /**
     * Creates a new {@link TracingScope} and makes it the current one. {@link TracingScope#end()}
     * must be called to properly close it.
     *
     * @return the current tracing scope
     */
    TracingScope makeCurrent();

    /**
     * Sets an attribute to this tracing span
     *
     * @param key the attribute key
     *
     * @param value the attribute value
     *
     * @return this instance, for method chaining
     */
    <T> TracingSpan setAttribute(TracingAttributeKey<T> key, T value);

    /**
     * Indicates that the operation completed successfully
     *
     * @return this instance, for method chaining
     */
    TracingSpan setStatusOk();

    /**
     * Sets this span status to error. followed by setting the attributes
     * {@link TracingAttributeKey#ERROR_TYPE} and {@link TracingAttributeKey#ERROR_MESSAGE}
     * according to the given {@code errorType} and {@code errorMessage} respectively.
     * <p>
     * If an error status was previously set it will be overridden.
     *
     * @param errorType the value for the {@link TracingAttributeKey#ERROR_TYPE} attribute; if
     *        {@code null}, will be ignored and the attribute won't be set
     *
     * @param errorMessage the value for the {@link TracingAttributeKey#ERROR_MESSAGE} attribute; if
     *        {@code null}, will be ignored and the attribute won't be set
     *
     * @return this instance, for method chaining
     *
     * @see TracingAttributeKey#ERROR
     * @see TracingAttributeKey#ERROR_TYPE
     * @see TracingAttributeKey#ERROR_MESSAGE
     */
    TracingSpan setStatusError(String errorType, String errorMessage);

    /**
     * Equivalent to invoking {@code setStatusError(exception.getClass().getCanonicalName(),
     * exception.getMessage())}.
     * <p>
     * In addition if the tracing level is {@link TracingLevel#TRACE TRACE}, adds to the span an
     * event describing the given exception.
     *
     * @param exception the exception for which the relevant attributes should be set in the span;
     *        can be {@code null}
     *
     * @return this instance, for method chaining
     *
     * @see TracingAttributeKey#ERROR
     * @see TracingAttributeKey#ERROR_TYPE
     * @see TracingAttributeKey#ERROR_MESSAGE
     */
    TracingSpan setStatusError(Throwable exception);

    /**
     * Sets the span's status to error, extracts the error type and error message from the given
     * {@code errorValue} and sets the span's error attributes.
     *
     * @param errorValue contains information about the error which should be recorded in the span
     *
     * @return this instance, for method chaining
     *
     * @see TracingAttributeKey#ERROR
     * @see TracingAttributeKey#ERROR_TYPE
     * @see TracingAttributeKey#ERROR_MESSAGE
     */
    TracingSpan setStatusError(ErrorValue errorValue);

    /**
     * If the tracing level is {@link TracingLevel#DEBUG DEBUG} or more
     * verbose, adds to the span an event with the given {@code name}.
     *
     * @param name
     *        the name of the event; if {@code null}, this method will
     *        do nothing
     *
     * @return this instance, for method chaining
     */
    TracingSpan addEvent(String name);

    /**
     * If the tracing level is {@link TracingLevel#DEBUG DEBUG} or more
     * verbose, adds to the span an event with the given {@code name}
     * and with an attribute with the given {@code key} and {@code value}
     *
     * @param name
     *        the name of the event; if {@code null}, this method will
     *        do nothing
     *
     * @param key
     *        the event, which this method will add to the span, will
     *        have an attribute with this key and with the given {@code
     *        value}; if the key is {@code null}, no attribute will be
     *        added to the event
     *
     * @param value
     *        the event, which this method will add to the span, will
     *        have an attribute with the given {@code key} whose value
     *        is this argument
     *
     * @return this instance, for method chaining
     */
    <T> TracingSpan addEvent(String name, TracingAttributeKey<T> key, T value);

    /**
     * Updates the name of this span
     *
     * @param newName the new name
     */
    TracingSpan updateName(String newName);

    /**
     * Ends the span
     *
     */
    void end();


    /**
     * For the purpose of trace context propagation between services, this
     * method injects this span into the given {@code carrier}. The carrier
     * is usually an HTTP request (e.g. an Apache HTTP client {@code
     * HttpPost} instance) and the setter usually adds headers to the request.
     *
     * @param carrier
     *        holds propagation fields. For example, an outgoing message or
     *        an HTTP request. If {@code null}, this method will do nothing.
     *
     * @param setter
     *        invoked for each propagation key; usually, the {@code carrier}
     *        is an HTTP request and this {@code setter} will add headers
     *        to the request. If {@code null}, this method will do nothing.
     *
     * @return this instance, for method chaining
     */
    // DEV NOTE: currently, we propagate only the *span* data; however, in
    // the future we may decide to also propagate some Baggage and other
    // stuff. In such case, we will need to make this method take either a
    // Context argument, or an additional Baggage argument
    <C> TracingSpan injectInto(C carrier, KeyValueSetter<C> setter);


    static interface KeyValueSetter<C> {
        /**
         * Sets a propagated field {@code key} to the given value {@code value} in the
         * {@code carrier}
         */
        void set(C carrier, String key, String value);
    }

}